/*
 * Decompiled with CFR 0.152.
 */
package org.apache.linkis.common.utils;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.apache.linkis.common.conf.CommonVars;
import org.apache.linkis.common.conf.CommonVars$;
import org.apache.linkis.common.exception.LinkisSecurityException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class SecurityUtils {
    private static final Logger logger = LoggerFactory.getLogger(SecurityUtils.class);
    private static final String COMMA = ",";
    private static final String EQUAL_SIGN = "=";
    private static final String AND_SYMBOL = "&";
    private static final String QUESTION_MARK = "?";
    private static final String REGEX_QUESTION_MARK = "\\?";
    private static final int JDBC_URL_ITEM_COUNT = 2;
    private static final CommonVars<String> MYSQL_SENSITIVE_PARAMS = CommonVars$.MODULE$.apply("linkis.mysql.sensitive.params", "allowLoadLocalInfile,autoDeserialize,allowLocalInfile,allowUrlInLocalInfile,#");
    private static final CommonVars<String> MYSQL_FORCE_PARAMS = CommonVars$.MODULE$.apply("linkis.mysql.force.params", "allowLoadLocalInfile=false&autoDeserialize=false&allowLocalInfile=false&allowUrlInLocalInfile=false");
    private static final CommonVars<String> MYSQL_STRONG_SECURITY_ENABLE = CommonVars$.MODULE$.apply("linkis.mysql.strong.security.enable", "false");
    private static final CommonVars<String> MYSQL_SECURITY_CHECK_ENABLE = CommonVars$.MODULE$.apply("linkis.mysql.security.check.enable", "true");
    private static final CommonVars<String> MYSQL_CONNECT_URL = CommonVars.apply("linkis.security.mysql.url.template", "jdbc:mysql://%s:%s/%s");
    private static final CommonVars<String> JDBC_MATCH_REGEX = CommonVars$.MODULE$.apply("linkis.mysql.jdbc.match.regex", "(?i)jdbc:(?i)(mysql)://([^:]+)(:[0-9]+)?(/[a-zA-Z0-9_-]*[\\.\\-]?)?");
    private static final String JDBC_MYSQL_PROTOCOL = "jdbc:mysql";
    private static final String BLACKLIST_REGEX = "autodeserialize|allowloadlocalinfile|allowurlinlocalinfile|allowloadlocalinfileinpath";

    public static void checkJdbcConnParams(String host, Integer port, String username, String password, String database, Map<String, Object> extraParams) {
        if (!Boolean.valueOf(MYSQL_SECURITY_CHECK_ENABLE.getValue()).booleanValue()) {
            return;
        }
        if (StringUtils.isAnyBlank((CharSequence[])new CharSequence[]{host, username})) {
            logger.error("Invalid mysql connection params: host: {}, username: {}, database: {}", new Object[]{host, username, database});
            throw new LinkisSecurityException(35000, "Invalid mysql connection params.");
        }
        String url = String.format(MYSQL_CONNECT_URL.getValue(), host.trim(), port, database.trim());
        SecurityUtils.checkUrl(url);
        SecurityUtils.checkParams(extraParams);
        SecurityUtils.checkUrlIsSafe(url);
    }

    public static void checkJdbcConnUrl(String url) {
        if (!Boolean.valueOf(MYSQL_SECURITY_CHECK_ENABLE.getValue()).booleanValue()) {
            return;
        }
        logger.info("jdbc url: {}", (Object)url);
        if (StringUtils.isBlank((CharSequence)url)) {
            throw new LinkisSecurityException(35000, "Invalid jdbc connection url.");
        }
        if (!url.toLowerCase().startsWith(JDBC_MYSQL_PROTOCOL)) {
            return;
        }
        String[] urlItems = url.split(REGEX_QUESTION_MARK);
        if (urlItems.length > 2) {
            throw new LinkisSecurityException(35000, "Invalid jdbc connection url.");
        }
        SecurityUtils.checkUrl(urlItems[0]);
        if (urlItems.length == 2) {
            Map<String, Object> params = SecurityUtils.parseMysqlUrlParamsToMap(urlItems[1]);
            SecurityUtils.checkParams(params);
        }
    }

    public static String getJdbcUrl(String url) {
        if (StringUtils.isBlank((CharSequence)url)) {
            return url;
        }
        if (!url.toLowerCase().startsWith(JDBC_MYSQL_PROTOCOL)) {
            return url;
        }
        String[] items = url.split(REGEX_QUESTION_MARK);
        String result = items[0];
        if (items.length == 2) {
            Map<String, Object> params = SecurityUtils.parseMysqlUrlParamsToMap(items[1]);
            SecurityUtils.appendMysqlForceParams(params);
            String paramUrl = SecurityUtils.parseParamsMapToMysqlParamUrl(params);
            result = result + QUESTION_MARK + paramUrl;
        }
        return result;
    }

    public static String appendMysqlForceParams(String url) {
        if (StringUtils.isBlank((CharSequence)url)) {
            return "";
        }
        if (!Boolean.valueOf(MYSQL_STRONG_SECURITY_ENABLE.getValue()).booleanValue()) {
            return url;
        }
        String extraParamString = MYSQL_FORCE_PARAMS.getValue();
        url = url.endsWith(QUESTION_MARK) ? url + extraParamString : (url.lastIndexOf(QUESTION_MARK) < 0 ? url + QUESTION_MARK + extraParamString : url + AND_SYMBOL + extraParamString);
        return url;
    }

    public static void appendMysqlForceParams(Map<String, Object> extraParams) {
        if (Boolean.valueOf(MYSQL_STRONG_SECURITY_ENABLE.getValue()).booleanValue()) {
            extraParams.putAll(SecurityUtils.parseMysqlUrlParamsToMap(MYSQL_FORCE_PARAMS.getValue()));
        }
    }

    public static String parseParamsMapToMysqlParamUrl(Map<String, Object> params) {
        if (params == null || params.isEmpty()) {
            return "";
        }
        return params.entrySet().stream().map(e -> String.join((CharSequence)EQUAL_SIGN, (CharSequence)e.getKey(), String.valueOf(e.getValue()))).collect(Collectors.joining(AND_SYMBOL));
    }

    public static void checkUrl(String url) {
        if (url != null && !url.toLowerCase().startsWith(JDBC_MYSQL_PROTOCOL)) {
            return;
        }
        Pattern regex = Pattern.compile(JDBC_MATCH_REGEX.getValue());
        Matcher matcher = regex.matcher(url);
        if (!matcher.matches()) {
            logger.info("Invalid mysql connection url: {}", (Object)url);
            throw new LinkisSecurityException(35000, "Invalid mysql connection url.");
        }
    }

    private static void checkParams(Map<String, Object> paramsMap) {
        if (paramsMap == null || paramsMap.isEmpty()) {
            return;
        }
        String paramUrl = SecurityUtils.parseParamsMapToMysqlParamUrl(paramsMap);
        try {
            paramUrl = URLDecoder.decode(paramUrl, "UTF-8");
        }
        catch (UnsupportedEncodingException e) {
            throw new LinkisSecurityException(35000, "mysql connection cul decode error: " + e);
        }
        Map<String, Object> newParamsMap = SecurityUtils.parseMysqlUrlParamsToMap(paramUrl);
        paramsMap.clear();
        paramsMap.putAll(newParamsMap);
        Iterator<Map.Entry<String, Object>> iterator = paramsMap.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<String, Object> entry = iterator.next();
            String key = entry.getKey();
            Object value = entry.getValue();
            if (StringUtils.isBlank((CharSequence)key) || value == null || StringUtils.isBlank((CharSequence)value.toString())) {
                logger.warn("Invalid parameter key or value is blank.");
                iterator.remove();
                continue;
            }
            if (!SecurityUtils.isNotSecurity(key, value.toString())) continue;
            logger.warn("Sensitive param : key={} and value={}", (Object)key, value);
            throw new LinkisSecurityException(35000, "Invalid mysql connection parameters: " + SecurityUtils.parseParamsMapToMysqlParamUrl(paramsMap));
        }
    }

    public static void checkUrlIsSafe(String url) {
        try {
            String lowercaseURL = url.toLowerCase();
            Pattern pattern = Pattern.compile(BLACKLIST_REGEX);
            Matcher matcher = pattern.matcher(lowercaseURL);
            StringBuilder foundKeywords = new StringBuilder();
            while (matcher.find()) {
                if (foundKeywords.length() > 0) {
                    foundKeywords.append(", ");
                }
                foundKeywords.append(matcher.group());
            }
            if (foundKeywords.length() > 0) {
                throw new LinkisSecurityException(35000, "url contains blacklisted characters: " + foundKeywords);
            }
        }
        catch (Exception e) {
            throw new LinkisSecurityException(35000, "error occurred during url security check: " + e);
        }
    }

    private static Map<String, Object> parseMysqlUrlParamsToMap(String paramsUrl) {
        if (StringUtils.isBlank((CharSequence)paramsUrl)) {
            return new LinkedHashMap<String, Object>();
        }
        String[] params = paramsUrl.split(AND_SYMBOL);
        LinkedHashMap<String, Object> map = new LinkedHashMap<String, Object>(params.length);
        for (String param : params) {
            String[] item = param.split(EQUAL_SIGN);
            if (item.length != 2) {
                logger.warn("mysql force param {} error.", (Object)param);
                continue;
            }
            map.put(item[0], item[1]);
        }
        return map;
    }

    private static boolean isNotSecurity(String key, String value) {
        String[] forceParams;
        boolean res = true;
        String sensitiveParamsStr = MYSQL_SENSITIVE_PARAMS.getValue();
        if (StringUtils.isBlank((CharSequence)sensitiveParamsStr)) {
            return false;
        }
        for (String forceParam : forceParams = sensitiveParamsStr.split(COMMA)) {
            if (!SecurityUtils.isNotSecurity(key, value, forceParam)) continue;
            res = false;
            break;
        }
        return !res;
    }

    private static boolean isNotSecurity(String key, String value, String param) {
        return key.toLowerCase().contains(param.toLowerCase()) || value.toLowerCase().contains(param.toLowerCase());
    }

    public static Properties getMysqlSecurityParams() {
        Properties properties = new Properties();
        properties.setProperty("allowLoadLocalInfile", "false");
        properties.setProperty("autoDeserialize", "false");
        properties.setProperty("allowLocalInfile", "false");
        properties.setProperty("allowUrlInLocalInfile", "false");
        return properties;
    }

    public static boolean containsRelativePath(String path) {
        if (path.startsWith("./") || path.contains("/./") || path.startsWith("../") || path.contains("/../")) {
            return true;
        }
        return path.startsWith(".\\") || path.contains("\\.\\") || path.startsWith("..\\") || path.contains("\\..\\");
    }
}

