#
# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.
from __future__ import annotations

from typing import TYPE_CHECKING

from airflow._shared.timezones import timezone
from airflow.models.taskreschedule import TaskReschedule
from airflow.ti_deps.deps.base_ti_dep import BaseTIDep
from airflow.utils.session import provide_session
from airflow.utils.state import TaskInstanceState

if TYPE_CHECKING:
    from collections.abc import Iterator

    from sqlalchemy.orm import Session

    from airflow.models.taskinstance import TaskInstance
    from airflow.ti_deps.dep_context import DepContext
    from airflow.ti_deps.deps.base_ti_dep import TIDepStatus


class ReadyToRescheduleDep(BaseTIDep):
    """Determines whether a task is ready to be rescheduled."""

    NAME = "Ready To Reschedule"
    IGNORABLE = True
    IS_TASK_DEP = True
    RESCHEDULEABLE_STATES = {TaskInstanceState.UP_FOR_RESCHEDULE, None}

    @provide_session
    def _get_dep_statuses(
        self,
        ti: TaskInstance,
        session: Session,
        dep_context: DepContext,
    ) -> Iterator[TIDepStatus]:
        """
        Determine whether a task is ready to be rescheduled.

        Only tasks in NONE or UP_FOR_RESCHEDULE state with at least one row in task_reschedule table are
        handled by this dependency class, otherwise this dependency is considered as passed.
        This dependency fails if the latest reschedule request's reschedule date is still
        in the future.
        """
        from airflow.models.mappedoperator import MappedOperator

        if dep_context.ignore_in_reschedule_period:
            yield self._passing_status(
                reason="The context specified that being in a reschedule period was permitted."
            )
            return

        if ti.state not in self.RESCHEDULEABLE_STATES:
            yield self._passing_status(
                reason="The task instance is not in State_UP_FOR_RESCHEDULE or NONE state."
            )
            return

        next_reschedule_date = session.scalar(
            TaskReschedule.stmt_for_task_instance(ti, descending=True)
            .with_only_columns(TaskReschedule.reschedule_date)
            .limit(1)
        )
        if not next_reschedule_date:
            # Because mapped sensors don't have the reschedule property, here's the last resort
            # and we need a slightly different passing reason
            if isinstance(ti.task, MappedOperator):
                yield self._passing_status(reason="The task is mapped and not in reschedule mode")
                return
            yield self._passing_status(reason="There is no reschedule request for this task instance.")
            return

        if (now := timezone.utcnow()) >= next_reschedule_date:
            yield self._passing_status(reason="Task instance id ready for reschedule.")
            return

        yield self._failing_status(
            reason=(
                "Task is not ready for reschedule yet but will be rescheduled automatically. "
                f"Current date is {now.isoformat()} and task will be "
                f"rescheduled at {next_reschedule_date.isoformat()}."
            )
        )
