/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.core.memory;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.ReadOnlyBufferException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import java.util.function.Function;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.flink.annotation.Internal;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.core.memory.MemoryUtils;
import org.apache.flink.util.Preconditions;
import sun.misc.Unsafe;

@Internal
public final class MemorySegment {
    public static final String CHECK_MULTIPLE_FREE_PROPERTY = "flink.tests.check-segment-multiple-free";
    private static final boolean checkMultipleFree = System.getProperties().containsKey("flink.tests.check-segment-multiple-free");
    private static final Unsafe UNSAFE = MemoryUtils.UNSAFE;
    private static final long BYTE_ARRAY_BASE_OFFSET = UNSAFE.arrayBaseOffset(byte[].class);
    private static final boolean LITTLE_ENDIAN = ByteOrder.nativeOrder() == ByteOrder.LITTLE_ENDIAN;
    @Nullable
    private final byte[] heapMemory;
    @Nullable
    private ByteBuffer offHeapBuffer;
    private long address;
    private final long addressLimit;
    private final int size;
    @Nullable
    private final Object owner;
    @Nullable
    private Runnable cleaner;
    private final boolean allowWrap;
    private final AtomicBoolean isFreedAtomic;

    MemorySegment(@Nonnull byte[] buffer, @Nullable Object owner) {
        this.heapMemory = buffer;
        this.offHeapBuffer = null;
        this.size = buffer.length;
        this.address = BYTE_ARRAY_BASE_OFFSET;
        this.addressLimit = this.address + (long)this.size;
        this.owner = owner;
        this.allowWrap = true;
        this.cleaner = null;
        this.isFreedAtomic = new AtomicBoolean(false);
    }

    MemorySegment(@Nonnull ByteBuffer buffer, @Nullable Object owner) {
        this(buffer, owner, true, null);
    }

    MemorySegment(@Nonnull ByteBuffer buffer, @Nullable Object owner, boolean allowWrap, @Nullable Runnable cleaner) {
        this.heapMemory = null;
        this.offHeapBuffer = buffer;
        this.size = buffer.capacity();
        this.address = MemoryUtils.getByteBufferAddress(buffer);
        this.addressLimit = this.address + (long)this.size;
        this.owner = owner;
        this.allowWrap = allowWrap;
        this.cleaner = cleaner;
        this.isFreedAtomic = new AtomicBoolean(false);
    }

    public int size() {
        return this.size;
    }

    @VisibleForTesting
    public boolean isFreed() {
        return this.address > this.addressLimit;
    }

    public void free() {
        if (this.isFreedAtomic.getAndSet(true)) {
            if (checkMultipleFree) {
                throw new IllegalStateException("MemorySegment can be freed only once!");
            }
        } else {
            this.address = this.addressLimit + 1L;
            this.offHeapBuffer = null;
            if (this.cleaner != null) {
                this.cleaner.run();
                this.cleaner = null;
            }
        }
    }

    public boolean isOffHeap() {
        return this.heapMemory == null;
    }

    public byte[] getArray() {
        if (this.heapMemory != null) {
            return this.heapMemory;
        }
        throw new IllegalStateException("Memory segment does not represent heap memory");
    }

    public ByteBuffer getOffHeapBuffer() {
        if (this.offHeapBuffer != null) {
            return this.offHeapBuffer;
        }
        throw new IllegalStateException("Memory segment does not represent off-heap buffer");
    }

    public long getAddress() {
        if (this.heapMemory == null) {
            return this.address;
        }
        throw new IllegalStateException("Memory segment does not represent off heap memory");
    }

    public ByteBuffer wrap(int offset, int length) {
        if (!this.allowWrap) {
            throw new UnsupportedOperationException("Wrap is not supported by this segment. This usually indicates that the underlying memory is unsafe, thus transferring of ownership is not allowed.");
        }
        return this.wrapInternal(offset, length);
    }

    private ByteBuffer wrapInternal(int offset, int length) {
        if (this.address <= this.addressLimit) {
            if (this.heapMemory != null) {
                return ByteBuffer.wrap(this.heapMemory, offset, length);
            }
            try {
                ByteBuffer wrapper = Preconditions.checkNotNull(this.offHeapBuffer).duplicate();
                wrapper.limit(offset + length);
                wrapper.position(offset);
                return wrapper;
            }
            catch (IllegalArgumentException e) {
                throw new IndexOutOfBoundsException();
            }
        }
        throw new IllegalStateException("segment has been freed");
    }

    @Nullable
    public Object getOwner() {
        return this.owner;
    }

    public byte get(int index) {
        long pos = this.address + (long)index;
        if (index >= 0 && pos < this.addressLimit) {
            return UNSAFE.getByte(this.heapMemory, pos);
        }
        if (this.address > this.addressLimit) {
            throw new IllegalStateException("segment has been freed");
        }
        throw new IndexOutOfBoundsException();
    }

    public void put(int index, byte b) {
        long pos = this.address + (long)index;
        if (index < 0 || pos >= this.addressLimit) {
            if (this.address > this.addressLimit) {
                throw new IllegalStateException("segment has been freed");
            }
            throw new IndexOutOfBoundsException();
        }
        UNSAFE.putByte(this.heapMemory, pos, b);
    }

    public void get(int index, byte[] dst) {
        this.get(index, dst, 0, dst.length);
    }

    public void put(int index, byte[] src) {
        this.put(index, src, 0, src.length);
    }

    public void get(int index, byte[] dst, int offset, int length) {
        if ((offset | length | offset + length | dst.length - (offset + length)) < 0) {
            throw new IndexOutOfBoundsException();
        }
        long pos = this.address + (long)index;
        if (index < 0 || pos > this.addressLimit - (long)length) {
            if (this.address > this.addressLimit) {
                throw new IllegalStateException("segment has been freed");
            }
            throw new IndexOutOfBoundsException(String.format("pos: %d, length: %d, index: %d, offset: %d", pos, length, index, offset));
        }
        long arrayAddress = BYTE_ARRAY_BASE_OFFSET + (long)offset;
        UNSAFE.copyMemory(this.heapMemory, pos, dst, arrayAddress, length);
    }

    public void put(int index, byte[] src, int offset, int length) {
        if ((offset | length | offset + length | src.length - (offset + length)) < 0) {
            throw new IndexOutOfBoundsException();
        }
        long pos = this.address + (long)index;
        if (index < 0 || pos > this.addressLimit - (long)length) {
            if (this.address > this.addressLimit) {
                throw new IllegalStateException("segment has been freed");
            }
            throw new IndexOutOfBoundsException();
        }
        long arrayAddress = BYTE_ARRAY_BASE_OFFSET + (long)offset;
        UNSAFE.copyMemory(src, arrayAddress, this.heapMemory, pos, length);
    }

    public boolean getBoolean(int index) {
        return this.get(index) != 0;
    }

    public void putBoolean(int index, boolean value) {
        this.put(index, (byte)(value ? 1 : 0));
    }

    public char getChar(int index) {
        long pos = this.address + (long)index;
        if (index >= 0 && pos <= this.addressLimit - 2L) {
            return UNSAFE.getChar(this.heapMemory, pos);
        }
        if (this.address > this.addressLimit) {
            throw new IllegalStateException("This segment has been freed.");
        }
        throw new IndexOutOfBoundsException();
    }

    public char getCharLittleEndian(int index) {
        if (LITTLE_ENDIAN) {
            return this.getChar(index);
        }
        return Character.reverseBytes(this.getChar(index));
    }

    public char getCharBigEndian(int index) {
        if (LITTLE_ENDIAN) {
            return Character.reverseBytes(this.getChar(index));
        }
        return this.getChar(index);
    }

    public void putChar(int index, char value) {
        long pos = this.address + (long)index;
        if (index < 0 || pos > this.addressLimit - 2L) {
            if (this.address > this.addressLimit) {
                throw new IllegalStateException("segment has been freed");
            }
            throw new IndexOutOfBoundsException();
        }
        UNSAFE.putChar(this.heapMemory, pos, value);
    }

    public void putCharLittleEndian(int index, char value) {
        if (LITTLE_ENDIAN) {
            this.putChar(index, value);
        } else {
            this.putChar(index, Character.reverseBytes(value));
        }
    }

    public void putCharBigEndian(int index, char value) {
        if (LITTLE_ENDIAN) {
            this.putChar(index, Character.reverseBytes(value));
        } else {
            this.putChar(index, value);
        }
    }

    public short getShort(int index) {
        long pos = this.address + (long)index;
        if (index >= 0 && pos <= this.addressLimit - 2L) {
            return UNSAFE.getShort(this.heapMemory, pos);
        }
        if (this.address > this.addressLimit) {
            throw new IllegalStateException("segment has been freed");
        }
        throw new IndexOutOfBoundsException();
    }

    public short getShortLittleEndian(int index) {
        if (LITTLE_ENDIAN) {
            return this.getShort(index);
        }
        return Short.reverseBytes(this.getShort(index));
    }

    public short getShortBigEndian(int index) {
        if (LITTLE_ENDIAN) {
            return Short.reverseBytes(this.getShort(index));
        }
        return this.getShort(index);
    }

    public void putShort(int index, short value) {
        long pos = this.address + (long)index;
        if (index < 0 || pos > this.addressLimit - 2L) {
            if (this.address > this.addressLimit) {
                throw new IllegalStateException("segment has been freed");
            }
            throw new IndexOutOfBoundsException();
        }
        UNSAFE.putShort(this.heapMemory, pos, value);
    }

    public void putShortLittleEndian(int index, short value) {
        if (LITTLE_ENDIAN) {
            this.putShort(index, value);
        } else {
            this.putShort(index, Short.reverseBytes(value));
        }
    }

    public void putShortBigEndian(int index, short value) {
        if (LITTLE_ENDIAN) {
            this.putShort(index, Short.reverseBytes(value));
        } else {
            this.putShort(index, value);
        }
    }

    public int getInt(int index) {
        long pos = this.address + (long)index;
        if (index >= 0 && pos <= this.addressLimit - 4L) {
            return UNSAFE.getInt(this.heapMemory, pos);
        }
        if (this.address > this.addressLimit) {
            throw new IllegalStateException("segment has been freed");
        }
        throw new IndexOutOfBoundsException();
    }

    public int getIntLittleEndian(int index) {
        if (LITTLE_ENDIAN) {
            return this.getInt(index);
        }
        return Integer.reverseBytes(this.getInt(index));
    }

    public int getIntBigEndian(int index) {
        if (LITTLE_ENDIAN) {
            return Integer.reverseBytes(this.getInt(index));
        }
        return this.getInt(index);
    }

    public void putInt(int index, int value) {
        long pos = this.address + (long)index;
        if (index < 0 || pos > this.addressLimit - 4L) {
            if (this.address > this.addressLimit) {
                throw new IllegalStateException("segment has been freed");
            }
            throw new IndexOutOfBoundsException();
        }
        UNSAFE.putInt(this.heapMemory, pos, value);
    }

    public void putIntLittleEndian(int index, int value) {
        if (LITTLE_ENDIAN) {
            this.putInt(index, value);
        } else {
            this.putInt(index, Integer.reverseBytes(value));
        }
    }

    public void putIntBigEndian(int index, int value) {
        if (LITTLE_ENDIAN) {
            this.putInt(index, Integer.reverseBytes(value));
        } else {
            this.putInt(index, value);
        }
    }

    public long getLong(int index) {
        long pos = this.address + (long)index;
        if (index >= 0 && pos <= this.addressLimit - 8L) {
            return UNSAFE.getLong(this.heapMemory, pos);
        }
        if (this.address > this.addressLimit) {
            throw new IllegalStateException("segment has been freed");
        }
        throw new IndexOutOfBoundsException();
    }

    public long getLongLittleEndian(int index) {
        if (LITTLE_ENDIAN) {
            return this.getLong(index);
        }
        return Long.reverseBytes(this.getLong(index));
    }

    public long getLongBigEndian(int index) {
        if (LITTLE_ENDIAN) {
            return Long.reverseBytes(this.getLong(index));
        }
        return this.getLong(index);
    }

    public void putLong(int index, long value) {
        long pos = this.address + (long)index;
        if (index < 0 || pos > this.addressLimit - 8L) {
            if (this.address > this.addressLimit) {
                throw new IllegalStateException("segment has been freed");
            }
            throw new IndexOutOfBoundsException();
        }
        UNSAFE.putLong(this.heapMemory, pos, value);
    }

    public void putLongLittleEndian(int index, long value) {
        if (LITTLE_ENDIAN) {
            this.putLong(index, value);
        } else {
            this.putLong(index, Long.reverseBytes(value));
        }
    }

    public void putLongBigEndian(int index, long value) {
        if (LITTLE_ENDIAN) {
            this.putLong(index, Long.reverseBytes(value));
        } else {
            this.putLong(index, value);
        }
    }

    public float getFloat(int index) {
        return Float.intBitsToFloat(this.getInt(index));
    }

    public float getFloatLittleEndian(int index) {
        return Float.intBitsToFloat(this.getIntLittleEndian(index));
    }

    public float getFloatBigEndian(int index) {
        return Float.intBitsToFloat(this.getIntBigEndian(index));
    }

    public void putFloat(int index, float value) {
        this.putInt(index, Float.floatToRawIntBits(value));
    }

    public void putFloatLittleEndian(int index, float value) {
        this.putIntLittleEndian(index, Float.floatToRawIntBits(value));
    }

    public void putFloatBigEndian(int index, float value) {
        this.putIntBigEndian(index, Float.floatToRawIntBits(value));
    }

    public double getDouble(int index) {
        return Double.longBitsToDouble(this.getLong(index));
    }

    public double getDoubleLittleEndian(int index) {
        return Double.longBitsToDouble(this.getLongLittleEndian(index));
    }

    public double getDoubleBigEndian(int index) {
        return Double.longBitsToDouble(this.getLongBigEndian(index));
    }

    public void putDouble(int index, double value) {
        this.putLong(index, Double.doubleToRawLongBits(value));
    }

    public void putDoubleLittleEndian(int index, double value) {
        this.putLongLittleEndian(index, Double.doubleToRawLongBits(value));
    }

    public void putDoubleBigEndian(int index, double value) {
        this.putLongBigEndian(index, Double.doubleToRawLongBits(value));
    }

    public void get(DataOutput out, int offset, int length) throws IOException {
        if (this.address <= this.addressLimit) {
            if (this.heapMemory != null) {
                out.write(this.heapMemory, offset, length);
            } else {
                while (length >= 8) {
                    out.writeLong(this.getLongBigEndian(offset));
                    offset += 8;
                    length -= 8;
                }
                while (length > 0) {
                    out.writeByte(this.get(offset));
                    ++offset;
                    --length;
                }
            }
        } else {
            throw new IllegalStateException("segment has been freed");
        }
    }

    public void put(DataInput in, int offset, int length) throws IOException {
        if (this.address <= this.addressLimit) {
            if (this.heapMemory != null) {
                in.readFully(this.heapMemory, offset, length);
            } else {
                while (length >= 8) {
                    this.putLongBigEndian(offset, in.readLong());
                    offset += 8;
                    length -= 8;
                }
                while (length > 0) {
                    this.put(offset, in.readByte());
                    ++offset;
                    --length;
                }
            }
        } else {
            throw new IllegalStateException("segment has been freed");
        }
    }

    /*
     * Enabled aggressive block sorting
     */
    public void get(int offset, ByteBuffer target, int numBytes) {
        if ((offset | numBytes | offset + numBytes) < 0) {
            throw new IndexOutOfBoundsException();
        }
        if (target.isReadOnly()) {
            throw new ReadOnlyBufferException();
        }
        int targetOffset = target.position();
        int remaining = target.remaining();
        if (remaining < numBytes) {
            throw new BufferOverflowException();
        }
        if (!target.isDirect()) {
            if (!target.hasArray()) throw new IllegalArgumentException("The target buffer is not direct, and has no array.");
            this.get(offset, target.array(), targetOffset + target.arrayOffset(), numBytes);
            target.position(targetOffset + numBytes);
            return;
        }
        long targetPointer = MemoryUtils.getByteBufferAddress(target) + (long)targetOffset;
        long sourcePointer = this.address + (long)offset;
        if (sourcePointer <= this.addressLimit - (long)numBytes) {
            UNSAFE.copyMemory(this.heapMemory, sourcePointer, null, targetPointer, numBytes);
            target.position(targetOffset + numBytes);
            return;
        }
        if (this.address <= this.addressLimit) throw new IndexOutOfBoundsException();
        throw new IllegalStateException("segment has been freed");
    }

    /*
     * Enabled aggressive block sorting
     */
    public void put(int offset, ByteBuffer source, int numBytes) {
        if ((offset | numBytes | offset + numBytes) < 0) {
            throw new IndexOutOfBoundsException();
        }
        int sourceOffset = source.position();
        int remaining = source.remaining();
        if (remaining < numBytes) {
            throw new BufferUnderflowException();
        }
        if (source.isDirect()) {
            long sourcePointer = MemoryUtils.getByteBufferAddress(source) + (long)sourceOffset;
            long targetPointer = this.address + (long)offset;
            if (targetPointer <= this.addressLimit - (long)numBytes) {
                UNSAFE.copyMemory(null, sourcePointer, this.heapMemory, targetPointer, numBytes);
                source.position(sourceOffset + numBytes);
                return;
            }
            if (this.address <= this.addressLimit) throw new IndexOutOfBoundsException();
            throw new IllegalStateException("segment has been freed");
        }
        if (source.hasArray()) {
            this.put(offset, source.array(), sourceOffset + source.arrayOffset(), numBytes);
            source.position(sourceOffset + numBytes);
            return;
        }
        int i = 0;
        while (i < numBytes) {
            this.put(offset++, source.get());
            ++i;
        }
    }

    public void copyTo(int offset, MemorySegment target, int targetOffset, int numBytes) {
        byte[] thisHeapRef = this.heapMemory;
        byte[] otherHeapRef = target.heapMemory;
        long thisPointer = this.address + (long)offset;
        long otherPointer = target.address + (long)targetOffset;
        if ((numBytes | offset | targetOffset) < 0 || thisPointer > this.addressLimit - (long)numBytes || otherPointer > target.addressLimit - (long)numBytes) {
            if (this.address > this.addressLimit) {
                throw new IllegalStateException("this memory segment has been freed.");
            }
            if (target.address > target.addressLimit) {
                throw new IllegalStateException("target memory segment has been freed.");
            }
            throw new IndexOutOfBoundsException(String.format("offset=%d, targetOffset=%d, numBytes=%d, address=%d, targetAddress=%d", offset, targetOffset, numBytes, this.address, target.address));
        }
        UNSAFE.copyMemory(thisHeapRef, thisPointer, otherHeapRef, otherPointer, numBytes);
    }

    public void copyToUnsafe(int offset, Object target, int targetPointer, int numBytes) {
        long thisPointer = this.address + (long)offset;
        if (thisPointer + (long)numBytes > this.addressLimit) {
            throw new IndexOutOfBoundsException(String.format("offset=%d, numBytes=%d, address=%d", offset, numBytes, this.address));
        }
        UNSAFE.copyMemory(this.heapMemory, thisPointer, target, targetPointer, numBytes);
    }

    public void copyFromUnsafe(int offset, Object source, int sourcePointer, int numBytes) {
        long thisPointer = this.address + (long)offset;
        if (thisPointer + (long)numBytes > this.addressLimit) {
            throw new IndexOutOfBoundsException(String.format("offset=%d, numBytes=%d, address=%d", offset, numBytes, this.address));
        }
        UNSAFE.copyMemory(source, sourcePointer, this.heapMemory, thisPointer, numBytes);
    }

    public int compare(MemorySegment seg2, int offset1, int offset2, int len) {
        while (len >= 8) {
            long l2;
            long l1 = this.getLongBigEndian(offset1);
            if (l1 != (l2 = seg2.getLongBigEndian(offset2))) {
                return l1 < l2 ^ l1 < 0L ^ l2 < 0L ? -1 : 1;
            }
            offset1 += 8;
            offset2 += 8;
            len -= 8;
        }
        while (len > 0) {
            int b2;
            int b1 = this.get(offset1) & 0xFF;
            int cmp = b1 - (b2 = seg2.get(offset2) & 0xFF);
            if (cmp != 0) {
                return cmp;
            }
            ++offset1;
            ++offset2;
            --len;
        }
        return 0;
    }

    public int compare(MemorySegment seg2, int offset1, int offset2, int len1, int len2) {
        int minLength = Math.min(len1, len2);
        int c = this.compare(seg2, offset1, offset2, minLength);
        return c == 0 ? len1 - len2 : c;
    }

    public void swapBytes(byte[] tempBuffer, MemorySegment seg2, int offset1, int offset2, int len) {
        if ((offset1 | offset2 | len | tempBuffer.length - len) >= 0) {
            long thisPos = this.address + (long)offset1;
            long otherPos = seg2.address + (long)offset2;
            if (thisPos <= this.addressLimit - (long)len && otherPos <= seg2.addressLimit - (long)len) {
                UNSAFE.copyMemory(this.heapMemory, thisPos, tempBuffer, BYTE_ARRAY_BASE_OFFSET, len);
                UNSAFE.copyMemory(seg2.heapMemory, otherPos, this.heapMemory, thisPos, len);
                UNSAFE.copyMemory(tempBuffer, BYTE_ARRAY_BASE_OFFSET, seg2.heapMemory, otherPos, len);
                return;
            }
            if (this.address > this.addressLimit) {
                throw new IllegalStateException("this memory segment has been freed.");
            }
            if (seg2.address > seg2.addressLimit) {
                throw new IllegalStateException("other memory segment has been freed.");
            }
        }
        throw new IndexOutOfBoundsException(String.format("offset1=%d, offset2=%d, len=%d, bufferSize=%d, address1=%d, address2=%d", offset1, offset2, len, tempBuffer.length, this.address, seg2.address));
    }

    public boolean equalTo(MemorySegment seg2, int offset1, int offset2, int length) {
        int i;
        for (i = 0; i <= length - 8; i += 8) {
            if (this.getLong(offset1 + i) == seg2.getLong(offset2 + i)) continue;
            return false;
        }
        while (i < length) {
            if (this.get(offset1 + i) != seg2.get(offset2 + i)) {
                return false;
            }
            ++i;
        }
        return true;
    }

    public byte[] getHeapMemory() {
        return this.heapMemory;
    }

    public <T> T processAsByteBuffer(Function<ByteBuffer, T> processFunction) {
        return Preconditions.checkNotNull(processFunction).apply(this.wrapInternal(0, this.size));
    }

    public void processAsByteBuffer(Consumer<ByteBuffer> processConsumer) {
        Preconditions.checkNotNull(processConsumer).accept(this.wrapInternal(0, this.size));
    }
}

