/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.om.snapshot;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.cache.CacheLoader;
import java.io.IOException;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import org.apache.hadoop.hdds.utils.Scheduler;
import org.apache.hadoop.hdds.utils.db.Table;
import org.apache.hadoop.ozone.om.OMMetadataManager;
import org.apache.hadoop.ozone.om.OMMetrics;
import org.apache.hadoop.ozone.om.OmSnapshot;
import org.apache.hadoop.ozone.om.exceptions.OMException;
import org.apache.hadoop.ozone.om.lock.FlatResource;
import org.apache.hadoop.ozone.om.lock.IOzoneManagerLock;
import org.apache.hadoop.ozone.om.lock.OMLockDetails;
import org.apache.hadoop.ozone.om.snapshot.ReferenceCounted;
import org.apache.hadoop.ozone.om.snapshot.ReferenceCountedCallback;
import org.apache.ozone.rocksdiff.RocksDBCheckpointDiffer;
import org.apache.ratis.util.BatchLogger;
import org.apache.ratis.util.TimeDuration;
import org.apache.ratis.util.function.UncheckedAutoCloseableSupplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SnapshotCache
implements ReferenceCountedCallback,
AutoCloseable {
    static final Logger LOG = LoggerFactory.getLogger(SnapshotCache.class);
    private static final long CACHE_WARNING_THROTTLE_INTERVAL_MS = 60000L;
    private final ConcurrentHashMap<UUID, ReferenceCounted<OmSnapshot>> dbMap = new ConcurrentHashMap();
    private final CacheLoader<UUID, OmSnapshot> cacheLoader;
    private final int cacheSizeLimit;
    private final Set<UUID> pendingEvictionQueue;
    private final Scheduler scheduler;
    private final IOzoneManagerLock lock;
    private static final String SNAPSHOT_CACHE_CLEANUP_SERVICE = "SnapshotCacheCleanupService";
    private final boolean compactNonSnapshotDiffTables;
    private final OMMetrics omMetrics;

    private boolean shouldCompactTable(String tableName) {
        return !RocksDBCheckpointDiffer.COLUMN_FAMILIES_TO_TRACK_IN_DAG.contains(tableName);
    }

    private void compactSnapshotDB(OmSnapshot snapshot) throws IOException {
        if (!this.compactNonSnapshotDiffTables) {
            return;
        }
        OMMetadataManager metadataManager = snapshot.getMetadataManager();
        for (Table table : metadataManager.getStore().listTables()) {
            if (!this.shouldCompactTable(table.getName())) continue;
            try {
                metadataManager.getStore().compactTable(table.getName());
            }
            catch (IOException e) {
                LOG.warn("Failed to compact table {} in snapshot {}: {}", new Object[]{table.getName(), snapshot.getSnapshotID(), e.getMessage()});
            }
        }
    }

    public SnapshotCache(CacheLoader<UUID, OmSnapshot> cacheLoader, int cacheSizeLimit, OMMetrics omMetrics, long cleanupInterval, boolean compactNonSnapshotDiffTables, IOzoneManagerLock lock) {
        this.cacheLoader = cacheLoader;
        this.cacheSizeLimit = cacheSizeLimit;
        this.omMetrics = omMetrics;
        this.lock = lock;
        this.pendingEvictionQueue = ConcurrentHashMap.newKeySet();
        this.compactNonSnapshotDiffTables = compactNonSnapshotDiffTables;
        if (cleanupInterval > 0L) {
            this.scheduler = new Scheduler(SNAPSHOT_CACHE_CLEANUP_SERVICE, true, 1);
            this.scheduler.scheduleWithFixedDelay(() -> {
                Void void_ = this.cleanup(false);
            }, cleanupInterval, cleanupInterval, TimeUnit.MILLISECONDS);
        } else {
            this.scheduler = null;
        }
    }

    @VisibleForTesting
    ConcurrentHashMap<UUID, ReferenceCounted<OmSnapshot>> getDbMap() {
        return this.dbMap;
    }

    public int size() {
        return this.dbMap.size();
    }

    public void invalidate(UUID key) {
        this.dbMap.compute(key, (k, v) -> {
            if (v == null) {
                LOG.debug("SnapshotId: '{}' does not exist in snapshot cache.", k);
            } else {
                try {
                    ((OmSnapshot)v.get()).close();
                }
                catch (IOException e) {
                    throw new IllegalStateException("Failed to close snapshotId: " + key, e);
                }
                this.omMetrics.decNumSnapshotCacheSize();
            }
            return null;
        });
    }

    public void invalidateAll() {
        for (UUID key : this.dbMap.keySet()) {
            this.invalidate(key);
        }
    }

    @Override
    public void close() {
        this.invalidateAll();
        if (this.scheduler != null) {
            this.scheduler.close();
        }
    }

    public UncheckedAutoCloseableSupplier<OmSnapshot> get(final UUID key) throws IOException {
        OMLockDetails lockDetails;
        if (this.size() > this.cacheSizeLimit) {
            BatchLogger.print((BatchLogger.Key)BatchLogKey.SNAPSHOT_CACHE_SIZE_EXCEEDED, (Object)"CacheSizeWarning", suffix -> LOG.warn("Snapshot cache size ({}) exceeds configured soft-limit ({}).{}", new Object[]{this.size(), this.cacheSizeLimit, suffix}));
        }
        if (!(lockDetails = this.lock.acquireReadLock((IOzoneManagerLock.Resource)FlatResource.SNAPSHOT_DB_LOCK, new String[]{key.toString()})).isLockAcquired()) {
            throw new OMException("Unable to acquire readlock on snapshot db with key " + key, OMException.ResultCodes.INTERNAL_ERROR);
        }
        final ReferenceCounted rcOmSnapshot = this.dbMap.compute(key, (k, v) -> {
            if (v == null) {
                LOG.info("Loading SnapshotId: '{}'", k);
                try {
                    v = new ReferenceCounted<OmSnapshot>((OmSnapshot)this.cacheLoader.load((Object)key), false, this);
                }
                catch (OMException omEx) {
                    if (!omEx.getResult().equals((Object)OMException.ResultCodes.FILE_NOT_FOUND)) {
                        throw new IllegalStateException(omEx);
                    }
                }
                catch (IOException ioEx) {
                    throw new IllegalStateException(ioEx);
                }
                catch (Exception ex) {
                    throw new IllegalStateException(ex);
                }
                this.omMetrics.incNumSnapshotCacheSize();
            }
            if (v != null) {
                v.incrementRefCount();
            }
            return v;
        });
        if (rcOmSnapshot == null) {
            this.lock.releaseReadLock((IOzoneManagerLock.Resource)FlatResource.SNAPSHOT_DB_LOCK, new String[]{key.toString()});
            throw new OMException("SnapshotId: '" + key + "' not found, or the snapshot is no longer active.", OMException.ResultCodes.FILE_NOT_FOUND);
        }
        return new UncheckedAutoCloseableSupplier<OmSnapshot>(){
            private final AtomicReference<Boolean> closed = new AtomicReference<Boolean>(false);

            public OmSnapshot get() {
                return (OmSnapshot)rcOmSnapshot.get();
            }

            public void close() {
                this.closed.updateAndGet(alreadyClosed -> {
                    if (!alreadyClosed.booleanValue()) {
                        rcOmSnapshot.decrementRefCount();
                        SnapshotCache.this.lock.releaseReadLock((IOzoneManagerLock.Resource)FlatResource.SNAPSHOT_DB_LOCK, new String[]{key.toString()});
                    }
                    return true;
                });
            }
        };
    }

    public void release(UUID key) {
        ReferenceCounted<OmSnapshot> val = this.dbMap.get(key);
        if (val == null) {
            throw new IllegalArgumentException("Key '" + key + "' does not " + "exist in cache.");
        }
        val.decrementRefCount();
    }

    public UncheckedAutoCloseableSupplier<OMLockDetails> lock() {
        return this.lock(() -> this.lock.acquireResourceWriteLock((IOzoneManagerLock.Resource)FlatResource.SNAPSHOT_DB_LOCK), () -> this.lock.releaseResourceWriteLock((IOzoneManagerLock.Resource)FlatResource.SNAPSHOT_DB_LOCK), () -> this.cleanup(true));
    }

    public UncheckedAutoCloseableSupplier<OMLockDetails> lock(UUID snapshotId) {
        return this.lock(() -> this.lock.acquireWriteLock((IOzoneManagerLock.Resource)FlatResource.SNAPSHOT_DB_LOCK, new String[]{snapshotId.toString()}), () -> this.lock.releaseWriteLock((IOzoneManagerLock.Resource)FlatResource.SNAPSHOT_DB_LOCK, new String[]{snapshotId.toString()}), () -> this.cleanup(snapshotId));
    }

    private OMLockDetails getEmptyOmLockDetails(OMLockDetails lockDetails) {
        return lockDetails.isLockAcquired() ? OMLockDetails.EMPTY_DETAILS_LOCK_ACQUIRED : OMLockDetails.EMPTY_DETAILS_LOCK_NOT_ACQUIRED;
    }

    private UncheckedAutoCloseableSupplier<OMLockDetails> lock(Supplier<OMLockDetails> lockFunction, Supplier<OMLockDetails> unlockFunction, Supplier<Void> cleanupFunction) {
        Supplier<OMLockDetails> emptyLockFunction = () -> this.getEmptyOmLockDetails((OMLockDetails)lockFunction.get());
        final Supplier<OMLockDetails> emptyUnlockFunction = () -> this.getEmptyOmLockDetails((OMLockDetails)unlockFunction.get());
        final AtomicReference<OMLockDetails> lockDetails = new AtomicReference<OMLockDetails>(emptyLockFunction.get());
        if (lockDetails.get().isLockAcquired()) {
            cleanupFunction.get();
            if (!this.dbMap.isEmpty()) {
                lockDetails.set(emptyUnlockFunction.get());
            }
        }
        return new UncheckedAutoCloseableSupplier<OMLockDetails>(){

            public void close() {
                lockDetails.updateAndGet(arg_0 -> 2.lambda$0((Supplier)emptyUnlockFunction, arg_0));
            }

            public OMLockDetails get() {
                return (OMLockDetails)lockDetails.get();
            }

            private static /* synthetic */ OMLockDetails lambda$0(Supplier supplier, OMLockDetails prevLock) {
                if (prevLock != null && prevLock.isLockAcquired()) {
                    return (OMLockDetails)supplier.get();
                }
                return prevLock;
            }
        };
    }

    private synchronized Void cleanup(boolean force) {
        if (force || this.dbMap.size() > this.cacheSizeLimit) {
            for (UUID evictionKey : this.pendingEvictionQueue) {
                this.cleanup(evictionKey);
            }
        }
        return null;
    }

    private synchronized Void cleanup(UUID evictionKey) {
        ReferenceCounted<OmSnapshot> snapshot = this.dbMap.get(evictionKey);
        if (snapshot != null && snapshot.getTotalRefCount() == 0L) {
            try {
                this.compactSnapshotDB(snapshot.get());
            }
            catch (IOException e) {
                LOG.warn("Failed to compact snapshot DB for snapshotId {}: {}", (Object)evictionKey, (Object)e.getMessage());
            }
        }
        this.dbMap.compute(evictionKey, (k, v) -> {
            this.pendingEvictionQueue.remove(k);
            if (v == null) {
                throw new IllegalStateException("SnapshotId '" + k + "' does not exist in cache. The RocksDB " + "instance of the Snapshot may not be closed properly.");
            }
            if (v.getTotalRefCount() > 0L) {
                LOG.debug("SnapshotId {} is still being referenced ({}), skipping its clean up.", k, (Object)v.getTotalRefCount());
                return v;
            }
            LOG.debug("Closing SnapshotId {}. It is not being referenced anymore.", k);
            try {
                ((OmSnapshot)v.get()).close();
            }
            catch (IOException ex) {
                throw new IllegalStateException("Error while closing snapshot DB.", ex);
            }
            this.omMetrics.decNumSnapshotCacheSize();
            return null;
        });
        return null;
    }

    @Override
    public void callback(ReferenceCounted referenceCounted) {
        if (referenceCounted.getTotalRefCount() == 0L) {
            this.pendingEvictionQueue.add(((OmSnapshot)referenceCounted.get()).getSnapshotID());
        }
    }

    long totalRefCount(UUID key) {
        return this.dbMap.containsKey(key) ? this.dbMap.get(key).getTotalRefCount() : 0L;
    }

    private static enum BatchLogKey implements BatchLogger.Key
    {
        SNAPSHOT_CACHE_SIZE_EXCEEDED;


        public TimeDuration getBatchDuration() {
            return TimeDuration.valueOf((long)60000L, (TimeUnit)TimeUnit.MILLISECONDS);
        }
    }

    public static enum Reason {
        FS_API_READ,
        SNAP_DIFF_READ,
        DEEP_CLEAN_WRITE,
        GARBAGE_COLLECTION_WRITE;

    }
}

