/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.container.balancer;

import java.time.Clock;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.container.ContainerInfo;
import org.apache.hadoop.hdds.scm.container.ContainerManager;
import org.apache.hadoop.hdds.scm.container.ContainerNotFoundException;
import org.apache.hadoop.hdds.scm.container.ContainerReplica;
import org.apache.hadoop.hdds.scm.container.ContainerReplicaNotFoundException;
import org.apache.hadoop.hdds.scm.container.common.helpers.MoveDataNodePair;
import org.apache.hadoop.hdds.scm.container.replication.ContainerHealthResult;
import org.apache.hadoop.hdds.scm.container.replication.ContainerReplicaOp;
import org.apache.hadoop.hdds.scm.container.replication.ContainerReplicaPendingOpsSubscriber;
import org.apache.hadoop.hdds.scm.container.replication.ReplicationManager;
import org.apache.hadoop.hdds.scm.node.NodeStatus;
import org.apache.hadoop.hdds.scm.node.states.NodeNotFoundException;
import org.apache.ratis.protocol.exceptions.NotLeaderException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class MoveManager
implements ContainerReplicaPendingOpsSubscriber {
    private static final Logger LOG = LoggerFactory.getLogger(MoveManager.class);
    private long moveTimeout = 3900000L;
    private long replicationTimeout = 3000000L;
    private final ReplicationManager replicationManager;
    private final ContainerManager containerManager;
    private final Clock clock;
    private final Map<ContainerID, MoveOperation> pendingMoves = new ConcurrentHashMap<ContainerID, MoveOperation>();

    public MoveManager(ReplicationManager replicationManager, ContainerManager containerManager) {
        this.replicationManager = replicationManager;
        this.containerManager = containerManager;
        this.clock = replicationManager.getClock();
    }

    public Map<ContainerID, MoveOperation> getPendingMove() {
        return this.pendingMoves;
    }

    void resetState() {
        this.pendingMoves.clear();
    }

    private void completeMove(ContainerID cid, MoveResult mr) {
        CompletableFuture<MoveResult> future;
        MoveOperation move = this.pendingMoves.remove(cid);
        if (move != null && (future = move.getResult()) != null && mr != null) {
            LOG.debug("Completing container move for container {} with result {}.", (Object)cid, (Object)mr);
            future.complete(mr);
        }
    }

    private void startMove(ContainerInfo containerInfo, DatanodeDetails src, DatanodeDetails tgt, CompletableFuture<MoveResult> ret) {
        MoveOperation move = this.pendingMoves.putIfAbsent(containerInfo.containerID(), new MoveOperation(ret, new MoveDataNodePair(src, tgt)));
        if (move == null) {
            try {
                this.sendReplicateCommand(containerInfo, tgt, src);
            }
            catch (Exception e) {
                LOG.error("Unable to schedule the replication command for container {}", (Object)containerInfo, (Object)e);
                ret.complete(MoveResult.FAIL_UNEXPECTED_ERROR);
                this.pendingMoves.remove(containerInfo.containerID());
            }
        } else {
            ret.complete(MoveResult.FAIL_CONTAINER_ALREADY_BEING_MOVED);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    CompletableFuture<MoveResult> move(ContainerID cid, DatanodeDetails src, DatanodeDetails tgt) throws ContainerNotFoundException, NodeNotFoundException, ContainerReplicaNotFoundException {
        ContainerInfo containerInfo;
        CompletableFuture<MoveResult> ret = new CompletableFuture<MoveResult>();
        for (DatanodeDetails dn : Arrays.asList(src, tgt)) {
            NodeStatus currentNodeStatus = this.replicationManager.getNodeStatus(dn);
            if (currentNodeStatus.getHealth() != HddsProtos.NodeState.HEALTHY) {
                ret.complete(MoveResult.REPLICATION_FAIL_NODE_UNHEALTHY);
                return ret;
            }
            if (currentNodeStatus.getOperationalState() == HddsProtos.NodeOperationalState.IN_SERVICE) continue;
            ret.complete(MoveResult.REPLICATION_FAIL_NODE_NOT_IN_SERVICE);
            return ret;
        }
        ContainerInfo containerInfo2 = containerInfo = this.containerManager.getContainer(cid);
        synchronized (containerInfo2) {
            Set<ContainerReplica> currentReplicas = this.containerManager.getContainerReplicas(cid);
            boolean srcExists = false;
            for (ContainerReplica r : currentReplicas) {
                if (r.getDatanodeDetails().equals((Object)src)) {
                    srcExists = true;
                }
                if (!r.getDatanodeDetails().equals((Object)tgt)) continue;
                ret.complete(MoveResult.REPLICATION_FAIL_EXIST_IN_TARGET);
                return ret;
            }
            if (!srcExists) {
                ret.complete(MoveResult.REPLICATION_FAIL_NOT_EXIST_IN_SOURCE);
                return ret;
            }
            ContainerHealthResult healthBeforeMove = this.replicationManager.getContainerReplicationHealth(containerInfo, currentReplicas);
            if (healthBeforeMove.getHealthState() != ContainerHealthResult.HealthState.HEALTHY) {
                ret.complete(MoveResult.REPLICATION_NOT_HEALTHY_BEFORE_MOVE);
                return ret;
            }
            List<ContainerReplicaOp> pendingOps = this.replicationManager.getPendingReplicationOps(cid);
            for (ContainerReplicaOp op : pendingOps) {
                if (op.getOpType() == ContainerReplicaOp.PendingOpType.ADD) {
                    ret.complete(MoveResult.REPLICATION_FAIL_INFLIGHT_REPLICATION);
                    return ret;
                }
                if (op.getOpType() != ContainerReplicaOp.PendingOpType.DELETE) continue;
                ret.complete(MoveResult.REPLICATION_FAIL_INFLIGHT_DELETION);
                return ret;
            }
            HddsProtos.LifeCycleState currentContainerStat = containerInfo.getState();
            if (currentContainerStat != HddsProtos.LifeCycleState.CLOSED) {
                ret.complete(MoveResult.REPLICATION_FAIL_CONTAINER_NOT_CLOSED);
                return ret;
            }
            Set<ContainerReplica> replicasAfterMove = this.createReplicaSetAfterMove(src, tgt, currentReplicas);
            ContainerHealthResult healthResult = this.replicationManager.getContainerReplicationHealth(containerInfo, replicasAfterMove);
            if (healthResult.getHealthState() != ContainerHealthResult.HealthState.HEALTHY) {
                ret.complete(MoveResult.REPLICATION_NOT_HEALTHY_AFTER_MOVE);
                return ret;
            }
            this.startMove(containerInfo, src, tgt, ret);
            LOG.debug("Processed a move request for container {}, from {} to {}", new Object[]{cid, src, tgt});
            return ret;
        }
    }

    private void notifyContainerOpCompleted(ContainerReplicaOp containerReplicaOp, ContainerID containerID) {
        MoveOperation move = this.pendingMoves.get(containerID);
        if (move != null) {
            MoveDataNodePair mdnp = move.getMoveDataNodePair();
            ContainerReplicaOp.PendingOpType opType = containerReplicaOp.getOpType();
            DatanodeDetails dn = containerReplicaOp.getTarget();
            if (opType.equals((Object)ContainerReplicaOp.PendingOpType.ADD) && mdnp.getTgt().equals((Object)dn)) {
                try {
                    this.handleSuccessfulAdd(containerID);
                }
                catch (ContainerNotFoundException | ContainerReplicaNotFoundException | NodeNotFoundException | NotLeaderException e) {
                    LOG.warn("Failed to handle successful Add for container {} being moved from source {} to target {}.", new Object[]{containerID, mdnp.getSrc(), mdnp.getTgt(), e});
                    move.getResult().complete(MoveResult.FAIL_UNEXPECTED_ERROR);
                }
            } else if (opType.equals((Object)ContainerReplicaOp.PendingOpType.DELETE) && mdnp.getSrc().equals((Object)dn)) {
                this.completeMove(containerID, MoveResult.COMPLETED);
            }
        }
    }

    private void notifyContainerOpExpired(ContainerReplicaOp containerReplicaOp, ContainerID containerID) {
        MoveOperation pair = this.pendingMoves.get(containerID);
        if (pair != null) {
            MoveDataNodePair mdnp = pair.getMoveDataNodePair();
            ContainerReplicaOp.PendingOpType opType = containerReplicaOp.getOpType();
            DatanodeDetails dn = containerReplicaOp.getTarget();
            if (opType.equals((Object)ContainerReplicaOp.PendingOpType.ADD) && mdnp.getTgt().equals((Object)dn)) {
                this.completeMove(containerID, MoveResult.REPLICATION_FAIL_TIME_OUT);
            } else if (opType.equals((Object)ContainerReplicaOp.PendingOpType.DELETE) && mdnp.getSrc().equals((Object)dn)) {
                this.completeMove(containerID, MoveResult.DELETION_FAIL_TIME_OUT);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void handleSuccessfulAdd(ContainerID cid) throws ContainerNotFoundException, ContainerReplicaNotFoundException, NodeNotFoundException, NotLeaderException {
        ContainerInfo containerInfo;
        MoveOperation moveOp = this.pendingMoves.get(cid);
        if (moveOp == null) {
            return;
        }
        MoveDataNodePair movePair = moveOp.getMoveDataNodePair();
        DatanodeDetails src = movePair.getSrc();
        DatanodeDetails tgt = movePair.getTgt();
        LOG.debug("Handling successful addition of Container {} from source {} to target {}.", new Object[]{cid, src, tgt});
        ContainerInfo containerInfo2 = containerInfo = this.containerManager.getContainer(cid);
        synchronized (containerInfo2) {
            Set<ContainerReplica> currentReplicas = this.containerManager.getContainerReplicas(cid);
            HashSet<ContainerReplica> futureReplicas = new HashSet<ContainerReplica>(currentReplicas);
            boolean found = futureReplicas.removeIf(r -> r.getDatanodeDetails().equals((Object)src));
            if (!found) {
                this.completeMove(cid, MoveResult.COMPLETED);
                return;
            }
            NodeStatus nodeStatus = this.replicationManager.getNodeStatus(src);
            if (nodeStatus.getOperationalState() != HddsProtos.NodeOperationalState.IN_SERVICE) {
                this.completeMove(cid, MoveResult.DELETION_FAIL_NODE_NOT_IN_SERVICE);
                return;
            }
            if (!nodeStatus.isHealthy()) {
                this.completeMove(cid, MoveResult.DELETION_FAIL_NODE_UNHEALTHY);
                return;
            }
            ContainerHealthResult healthResult = this.replicationManager.getContainerReplicationHealth(containerInfo, futureReplicas);
            if (healthResult.getHealthState() == ContainerHealthResult.HealthState.HEALTHY) {
                this.sendDeleteCommand(containerInfo, src, moveOp.getMoveStartTime());
            } else {
                LOG.info("Cannot remove source replica as the container health would be {}", (Object)healthResult.getHealthState());
                this.completeMove(cid, MoveResult.DELETE_FAIL_POLICY);
            }
        }
    }

    private Set<ContainerReplica> createReplicaSetAfterMove(DatanodeDetails src, DatanodeDetails tgt, Set<ContainerReplica> existing) {
        HashSet<ContainerReplica> replicas = new HashSet<ContainerReplica>(existing);
        ContainerReplica srcReplica = null;
        for (ContainerReplica r : replicas) {
            if (!r.getDatanodeDetails().equals((Object)src)) continue;
            srcReplica = r;
            break;
        }
        if (srcReplica == null) {
            throw new IllegalArgumentException("The source replica is not present");
        }
        replicas.remove(srcReplica);
        replicas.add(srcReplica.toBuilder().setDatanodeDetails(tgt).build());
        return replicas;
    }

    private void sendReplicateCommand(ContainerInfo containerInfo, DatanodeDetails tgt, DatanodeDetails src) throws ContainerReplicaNotFoundException, ContainerNotFoundException, NotLeaderException {
        int replicaIndex = this.getContainerReplicaIndex(containerInfo.containerID(), src);
        long now = this.clock.millis();
        this.replicationManager.sendLowPriorityReplicateContainerCommand(containerInfo, replicaIndex, src, tgt, now + this.replicationTimeout);
        this.pendingMoves.get(containerInfo.containerID()).setMoveStartTime(now);
    }

    private void sendDeleteCommand(ContainerInfo containerInfo, DatanodeDetails datanode, long moveStartTime) throws ContainerReplicaNotFoundException, ContainerNotFoundException, NotLeaderException {
        int replicaIndex = this.getContainerReplicaIndex(containerInfo.containerID(), datanode);
        this.replicationManager.sendDeleteCommand(containerInfo, replicaIndex, datanode, true, moveStartTime + this.moveTimeout);
    }

    private int getContainerReplicaIndex(ContainerID id, DatanodeDetails dn) throws ContainerNotFoundException, ContainerReplicaNotFoundException {
        Set<ContainerReplica> replicas = this.containerManager.getContainerReplicas(id);
        return replicas.stream().filter(r -> r.getDatanodeDetails().equals((Object)dn)).findFirst().orElseThrow(() -> new ContainerReplicaNotFoundException(id, dn)).getReplicaIndex();
    }

    @Override
    public void opCompleted(ContainerReplicaOp op, ContainerID containerID, boolean timedOut) {
        if (timedOut) {
            this.notifyContainerOpExpired(op, containerID);
        } else {
            this.notifyContainerOpCompleted(op, containerID);
        }
    }

    void setMoveTimeout(long moveTimeout) {
        this.moveTimeout = moveTimeout;
    }

    void setReplicationTimeout(long replicationTimeout) {
        this.replicationTimeout = replicationTimeout;
    }

    static class MoveOperation {
        private CompletableFuture<MoveResult> result;
        private MoveDataNodePair moveDataNodePair;
        private long moveStartTime;

        MoveOperation(CompletableFuture<MoveResult> result, MoveDataNodePair srcTgt) {
            this.result = result;
            this.moveDataNodePair = srcTgt;
        }

        public CompletableFuture<MoveResult> getResult() {
            return this.result;
        }

        public MoveDataNodePair getMoveDataNodePair() {
            return this.moveDataNodePair;
        }

        public long getMoveStartTime() {
            return this.moveStartTime;
        }

        public void setResult(CompletableFuture<MoveResult> result) {
            this.result = result;
        }

        public void setMoveDataNodePair(MoveDataNodePair srcTgt) {
            this.moveDataNodePair = srcTgt;
        }

        public void setMoveStartTime(long time) {
            this.moveStartTime = time;
        }
    }

    public static enum MoveResult {
        COMPLETED,
        FAIL_LEADER_NOT_READY,
        REPLICATION_FAIL_NOT_EXIST_IN_SOURCE,
        REPLICATION_FAIL_EXIST_IN_TARGET,
        REPLICATION_FAIL_CONTAINER_NOT_CLOSED,
        REPLICATION_FAIL_INFLIGHT_DELETION,
        REPLICATION_FAIL_INFLIGHT_REPLICATION,
        REPLICATION_FAIL_TIME_OUT,
        REPLICATION_FAIL_NODE_NOT_IN_SERVICE,
        REPLICATION_FAIL_NODE_UNHEALTHY,
        DELETION_FAIL_TIME_OUT,
        DELETION_FAIL_NODE_NOT_IN_SERVICE,
        DELETION_FAIL_NODE_UNHEALTHY,
        DELETE_FAIL_POLICY,
        REPLICATION_NOT_HEALTHY_BEFORE_MOVE,
        REPLICATION_NOT_HEALTHY_AFTER_MOVE,
        FAIL_CONTAINER_ALREADY_BEING_MOVED,
        FAIL_UNEXPECTED_ERROR;

    }
}

