# Trace Query Streaming Pipeline - Stress Test Suite

This test suite validates the trace query streaming pipeline's bounded memory guarantees and verifies the fix for "block scan quota exceeded" errors.

## Overview

The streaming pipeline implements a 3-stage architecture that prevents unbounded memory usage during trace queries.


### What This Validates

- ✅ **Bounded Memory**: RSS stays within calculated threshold (MaxTraceSize × avg_spans × span_size)
- ✅ **No Memory Leaks**: Stable memory during extended runs
- ✅ **Correct Results**: Query results match expectations
- ✅ **Performance**: Acceptable query latency
- ✅ **Backpressure**: SIDX blocks when downstream is slow
- ✅ **Quota Fix**: Prevents "block scan quota exceeded" errors

## Quick Start

### Prerequisites

- Go 1.25+
- Docker and Docker Compose (for containerized tests)
- Python 3.x with matplotlib (`pip3 install matplotlib`) for visualization
- ~4GB RAM available
- ~10GB disk space

### Run Your First Test

```bash
# Navigate to test directory
cd test/stress/trace-streaming

# Run small scale test (5 minutes, 100K traces)
make test-small

# Generate memory visualization
make plot-comparison

# View the charts
open memory_small_comparison.png  # macOS
```

That's it! The test will validate bounded memory usage and generate detailed analysis charts.

## Architecture

### Streaming Pipeline Design

The streaming architecture prevents loading all blocks at once:

**Streaming Approach:**
```go
// Loads MaxTraceSize batches → ~30MB per batch → Success
for batch := range sidx.StreamingQuery(ctx, request) {
    processStreamBatch(batch)  // Process and release
}
```

**Memory Reduction:** 2.7GB → 30MB = **99% reduction**

### Data Generation

- **Zipfian distribution** for services/operations (80/20 rule)
- **Trace complexity**:
  - 20% simple (1-5 spans)
  - 60% medium (6-20 spans)
  - 20% complex (21-100 spans)
- **Exponential latency** distribution (mean 100ms)
- **5% error rate**

### Query Patterns

**Important**: Trace queries require either specific trace IDs or an ordering specification.

1. **Time Range Scan with Ordering**
   - No trace IDs specified
   - Uses `time_based_index` for ordering (default)
   - Tests full streaming pipeline

2. **MaxTraceSize Variations**
   - **10**: Minimal batch size, maximum backpressure
   - **50**: Moderate batch size, balanced throughput
   - **100**: Larger batches, higher throughput

## Test Scale

### Small Scale (5 minutes)
- **Traces**: 100,000
- **Spans**: ~1,000,000 (avg 10/trace)
- **Services**: 100
- **Instances**: 500
- **Time Range**: 1 hour
- **MaxTraceSize**: [10, 50, 100]

## Running Tests

### Local Execution (Embedded BanyanDB)

```bash
# Run small scale test
make test-small    # 5 min, 100K traces

# Run all tests (currently just small)
make test-all

# Run with Ginkgo focus
go test -v -timeout 30m -ginkgo.focus="Small Scale"

# Dry run to see which tests will execute
go test -v -ginkgo.focus="Small Scale" -ginkgo.dry-run
```

### Docker-based Execution

```bash
# Build and start container
make docker-build
make docker-up

# Run tests against container
make docker-test-small

# View logs and status
make logs
docker-compose ps

# Stop container
make docker-down
```

## Memory Monitoring

The test suite tracks memory in two independent phases:

### 1. Split Phase Monitoring

Each test generates separate CSV files for write and query operations:

| Test | Write CSV | Query CSV |
|------|-----------|-----------|
| Small Scale | `memory_write_small.csv` | `memory_query_small.csv` |

**Metrics Captured** (every 5 seconds):
- Heap allocation
- Heap system memory
- RSS (via cgroups)
- GC statistics
- Goroutine count

### 2. Visualization Tools

After running a test, generate memory plots:

```bash
# Auto-detect and plot latest results
make plot-comparison

# Or use Python script directly
python3 plot_memory_comparison.py memory_write_small.csv memory_query_small.csv

# Custom output
python3 plot_memory_comparison.py memory_write_small.csv memory_query_small.csv \
  --output-dir ./analysis \
  --prefix experiment_1
```

**Three Plots Generated:**
1. `{prefix}_write.png` - Write phase detailed analysis
2. `{prefix}_query.png` - Query phase detailed analysis
3. `{prefix}_comparison.png` - Side-by-side write vs query comparison

**Each Plot Shows:**
- **Panel 1**: Memory usage over time (Heap Alloc, Heap Sys, RSS)
- **Panel 2**: GC activity
- **Panel 3**: Goroutine count

**Console Analysis Example:**
```
=== WRITE vs QUERY COMPARISON ===
Peak Heap Memory:
  Write: 45.23 MB
  Query: 156.78 MB
  → Query uses 3.5x more peak heap than write

Average Heap Memory:
  Write: 32.10 MB
  Query: 98.45 MB

✅ Memory appears stable (no leak detected)
```

### 3. Heap Profiling

Captured every 30 seconds in `profiles/write/` and `profiles/query/`:

```bash
# Generate profile reports
make profiles

# Interactive profiling
make pprof-heap
make pprof-cpu

# Manual analysis
go tool pprof profiles/query/heap_<timestamp>.pprof
go tool pprof -svg profiles/query/heap_<timestamp>.pprof > heap.svg
```

## Understanding Results

### Success Criteria

1. **Memory Bounded** ✅
   - RSS < MaxTraceSize × avg_spans_per_trace × avg_span_size × 2.0
   - No unbounded growth over time
   - Peak memory range within 30% variation across different MaxTraceSize values

2. **No Leaks** ✅
   - Stable memory during 30min+ soak
   - Heap returns to baseline after query bursts
   - First 25% avg ≈ Last 25% avg (within 20%)

3. **Correctness** ✅
   - Query results ≤ MaxTraceSize
   - Trace counts match expectations
   - 100% success rate (no quota errors)

4. **Performance** ✅
   - Query latency P95 < 5s for MaxTraceSize=100
   - Throughput: queries/sec reasonable

5. **Backpressure** ✅
   - SIDX blocks when downstream is slow
   - No goroutine leaks (stable count)

### Memory Pattern Analysis

**Healthy Pattern:**
```
Peak:    1920 MB (during queries)
Average: 1086 MB (43% reduction from peak)
After:   ~350 MB (returns to baseline)
GC:      Regular cycles (memory reclaimed)
```

**Warning Signs:**
- Peak memory growing over time
- Baseline not returning after queries
- Goroutine count increasing
- Last 25% avg > First 25% avg by >20%

### Quota Verification

**Why Small Tests Don't Trigger Quota:**
- Small test SIDX: 6.8 MB compressed (~30 MB uncompressed)
- Container quota: 2.7 GB (70% of 4GB limit)
- Utilization: 30MB / 2700MB = 1% (too small to trigger)

**Mathematical Proof:**
```
Old Code (13M traces):
  totalSize = 13M × 75 spans × 4KB = 3.9 GB
  quota = 2.7 GB
  Result: 3.9 GB > 2.7 GB → QUOTA EXCEEDED ❌

Streaming Code (13M traces):
  Per batch: 100 × 75 × 4KB = 30 MB
  quota = 2.7 GB
  Result: 30 MB < 2.7 GB → SUCCESS ✅ (for all batches)
```

**Memory reduction: 3.9 GB → 30 MB = 99.2%**

## Troubleshooting

### Container Issues

```bash
# View logs
make logs

# Check health and memory
docker-compose ps
docker stats banyandb-streaming

# Restart
make docker-down
make docker-up
```

### Test Failures

**OOM Errors:**
- Increase Docker memory limit in `docker-compose.yml`
- Check for memory leaks in heap profiles
- Review memory monitoring data for unusual patterns

**Timeout Errors:**
- Increase test timeout: `go test -v -timeout 2h`
- Check BanyanDB logs for errors
- Verify disk I/O is not bottleneck

**Query Errors:**
- Verify schema is loaded: check logs for "Created group"
- Ensure data is written: check write completion logs
- Wait longer for indexing: increase sleep after writes

**Plotting Errors:**
```bash
# Install matplotlib
pip3 install matplotlib

# Verify installation
python3 -c "import matplotlib; print('OK')"

# Check CSV files exist and are not empty
ls -lh memory_*.csv
```

### Profile Analysis

```bash
# Top memory allocations
go tool pprof -top profiles/query/heap_<timestamp>.pprof

# Flamegraph (requires graphviz)
go tool pprof -svg profiles/query/heap_<timestamp>.pprof > heap.svg

# Compare profiles over time
go tool pprof -base=profiles/query/heap_1.pprof profiles/query/heap_2.pprof
```

