/*
   This file is part of TALER
   Copyright (C) 2023-2025 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file exchangedb/pg_do_withdraw.c
 * @brief Implementation of the do_withdraw function for Postgres
 * @author Özgür Kesim
 */
#include "platform.h"
#include "taler_error_codes.h"
#include "taler_dbevents.h"
#include "taler_exchangedb_plugin.h"
#include "taler_pq_lib.h"
#include "pg_do_withdraw.h"
#include "pg_helper.h"
#include <gnunet/gnunet_time_lib.h>


enum GNUNET_DB_QueryStatus
TEH_PG_do_withdraw (
  void *cls,
  const struct TALER_EXCHANGEDB_Withdraw *withdraw,
  const struct GNUNET_TIME_Timestamp *timestamp,
  bool *balance_ok,
  struct TALER_Amount *reserve_balance,
  bool *age_ok,
  uint16_t *required_age,
  uint32_t *reserve_birthday,
  bool *idempotent,
  uint16_t *noreveal_index,
  bool *nonce_reuse)
{
  struct PostgresClosure *pg = cls;
  struct GNUNET_TIME_Timestamp gc;
  struct GNUNET_PQ_QueryParam params[] = {
    TALER_PQ_query_param_amount (pg->conn,
                                 &withdraw->amount_with_fee),
    GNUNET_PQ_query_param_auto_from_type (&withdraw->reserve_pub),
    GNUNET_PQ_query_param_auto_from_type (&withdraw->reserve_sig),
    GNUNET_PQ_query_param_timestamp (timestamp),
    GNUNET_PQ_query_param_timestamp (&gc),
    GNUNET_PQ_query_param_auto_from_type (&withdraw->planchets_h),
    (withdraw->age_proof_required)
    ? GNUNET_PQ_query_param_uint16 (&withdraw->max_age)
    : GNUNET_PQ_query_param_null (),
    (withdraw->age_proof_required)
    ? GNUNET_PQ_query_param_uint16 (&withdraw->noreveal_index)
    : GNUNET_PQ_query_param_null (),
    (withdraw->age_proof_required)
    ? GNUNET_PQ_query_param_auto_from_type (&withdraw->selected_h)
    : GNUNET_PQ_query_param_null (),
    GNUNET_PQ_query_param_array_uint64 (withdraw->num_coins,
                                        withdraw->denom_serials,
                                        pg->conn),
    TALER_PQ_query_param_array_blinded_denom_sig (withdraw->num_coins,
                                                  withdraw->denom_sigs,
                                                  pg->conn),
    (withdraw->no_blinding_seed)
    ? GNUNET_PQ_query_param_null ()
    : GNUNET_PQ_query_param_auto_from_type (&withdraw->blinding_seed),
    (withdraw->no_blinding_seed)
    ? GNUNET_PQ_query_param_null ()
    : TALER_PQ_query_param_array_cs_r_pub (withdraw->num_cs_r_values,
                                           withdraw->cs_r_values,
                                           pg->conn),
    (withdraw->no_blinding_seed)
    ? GNUNET_PQ_query_param_null ()
    : GNUNET_PQ_query_param_uint64 (&withdraw->cs_r_choices),
    GNUNET_PQ_query_param_end
  };
  bool reserve_found;
  bool no_noreveal_index;
  struct GNUNET_PQ_ResultSpec rs[] = {
    GNUNET_PQ_result_spec_bool ("out_reserve_found",
                                &reserve_found),
    GNUNET_PQ_result_spec_bool ("out_balance_ok",
                                balance_ok),
    TALER_PQ_RESULT_SPEC_AMOUNT ("out_reserve_balance",
                                 reserve_balance),
    GNUNET_PQ_result_spec_bool ("out_age_ok",
                                age_ok),
    GNUNET_PQ_result_spec_uint16 ("out_required_age",
                                  required_age),
    GNUNET_PQ_result_spec_uint32 ("out_reserve_birthday",
                                  reserve_birthday),
    GNUNET_PQ_result_spec_bool ("out_idempotent",
                                idempotent),
    GNUNET_PQ_result_spec_allow_null (
      GNUNET_PQ_result_spec_uint16 ("out_noreveal_index",
                                    noreveal_index),
      &no_noreveal_index),
    GNUNET_PQ_result_spec_bool ("out_nonce_reuse",
                                nonce_reuse),
    GNUNET_PQ_result_spec_end
  };
  enum GNUNET_DB_QueryStatus qs;

  GNUNET_assert ((withdraw->no_blinding_seed)  ==
                 (0 == withdraw->num_cs_r_values));

  gc = GNUNET_TIME_absolute_to_timestamp (
    GNUNET_TIME_absolute_add (timestamp->abs_time,
                              pg->legal_reserve_expiration_time));
  PREPARE (pg,
           "call_withdraw",
           "SELECT "
           " out_reserve_found"
           ",out_balance_ok"
           ",out_reserve_balance"
           ",out_age_ok"
           ",out_required_age"
           ",out_reserve_birthday"
           ",out_idempotent"
           ",out_noreveal_index"
           ",out_nonce_reuse"
           " FROM exchange_do_withdraw"
           " ($1,$2,$3,$4,$5,$6,$7,$8,$9,$10,$11,$12,$13,$14);");
  qs = GNUNET_PQ_eval_prepared_singleton_select (pg->conn,
                                                 "call_withdraw",
                                                 params,
                                                 rs);
  GNUNET_PQ_cleanup_query_params_closures (params);

  if (0 > qs)
    return qs;
  if (! reserve_found)
    return GNUNET_DB_STATUS_SUCCESS_NO_RESULTS;
  if ((withdraw->age_proof_required) &&
      (idempotent && no_noreveal_index))
    GNUNET_break (0);
  return qs;
}
