/*
   This file is part of TALER
   Copyright (C) 2025 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file exchangedb/pg_do_refresh.c
 * @brief Implementation of the do_refresh function for Postgres
 * @author Özgür Kesim
 */
#include "taler/platform.h"
#include "taler/taler_error_codes.h"
#include "taler/taler_dbevents.h"
#include "taler/taler_exchangedb_plugin.h"
#include "taler/taler_pq_lib.h"
#include "pg_do_refresh.h"
#include "pg_helper.h"


enum GNUNET_DB_QueryStatus
TEH_PG_do_refresh (
  void *cls,
  struct TALER_EXCHANGEDB_Refresh_vDOLDPLUS *refresh,
  const struct GNUNET_TIME_Timestamp *timestamp,
  bool *found,
  uint32_t *noreveal_index,
  bool *zombie_required,
  bool *nonce_reuse,
  bool *balance_ok,
  struct TALER_Amount *coin_balance)
{
  struct PostgresClosure *pg = cls;
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_auto_from_type (&refresh->rc),
    GNUNET_PQ_query_param_timestamp (timestamp),
    GNUNET_PQ_query_param_auto_from_type (&refresh->refresh_seed), /* 3 */
    (refresh->is_v27_refresh)
        ? GNUNET_PQ_query_param_null ()
        : GNUNET_PQ_query_param_array_auto_from_type (refresh->num_coins,
                                                      refresh->transfer_pubs,
                                                      pg->conn),
    GNUNET_PQ_query_param_auto_from_type (&refresh->planchets_h),
    TALER_PQ_query_param_amount (pg->conn, /* 6 */
                                 &refresh->amount_with_fee),
    (refresh->no_blinding_seed)
        ? GNUNET_PQ_query_param_null ()
        : GNUNET_PQ_query_param_auto_from_type (&refresh->blinding_seed),
    (0 < refresh->num_cs_r_values)
        ? TALER_PQ_query_param_array_cs_r_pub (refresh->num_cs_r_values,
                                               refresh->cs_r_values,
                                               pg->conn)
        : GNUNET_PQ_query_param_null (),
    (0 < refresh->num_cs_r_values)
        ? GNUNET_PQ_query_param_uint64 (&refresh->cs_r_choices) /* 9 */
        : GNUNET_PQ_query_param_null (),
    GNUNET_PQ_query_param_auto_from_type (&refresh->selected_h),
    TALER_PQ_query_param_array_blinded_denom_sig (refresh->num_coins,
                                                  refresh->denom_sigs,
                                                  pg->conn),
    GNUNET_PQ_query_param_array_uint64 (refresh->num_coins, /* 12 */
                                        refresh->denom_serials,
                                        pg->conn),
    GNUNET_PQ_query_param_auto_from_type (&refresh->coin.coin_pub),
    GNUNET_PQ_query_param_auto_from_type (&refresh->coin_sig),
    GNUNET_PQ_query_param_uint32 (&refresh->noreveal_index), /* 15 */
    GNUNET_PQ_query_param_bool (*zombie_required),
    GNUNET_PQ_query_param_end
  };
  bool coin_found;
  bool no_noreveal_index;
  bool no_coin_balance;
  struct GNUNET_PQ_ResultSpec rs[] = {
    GNUNET_PQ_result_spec_bool ("coin_found",
                                &coin_found),
    GNUNET_PQ_result_spec_bool ("balance_ok",
                                balance_ok),
    GNUNET_PQ_result_spec_bool ("zombie_required",
                                zombie_required),
    GNUNET_PQ_result_spec_bool ("nonce_reuse",
                                nonce_reuse),
    GNUNET_PQ_result_spec_bool ("found",
                                found),
    GNUNET_PQ_result_spec_allow_null (
      GNUNET_PQ_result_spec_uint32 ("noreveal_index",
                                    noreveal_index),
      &no_noreveal_index),
    GNUNET_PQ_result_spec_allow_null (
      TALER_PQ_RESULT_SPEC_AMOUNT ("coin_balance",
                                   coin_balance),
      &no_coin_balance),
    GNUNET_PQ_result_spec_end
  };
  enum GNUNET_DB_QueryStatus qs;

  PREPARE (pg,
           "call_refresh",
           "SELECT "
           " out_coin_found AS coin_found"
           ",out_balance_ok AS balance_ok"
           ",out_zombie_bad AS zombie_required"
           ",out_nonce_reuse AS nonce_reuse"
           ",out_idempotent AS found"
           ",out_noreveal_index AS noreveal_index"
           ",out_coin_balance AS coin_balance"
           " FROM exchange_do_refresh"
           " ($1,$2,$3,$4,$5,$6,$7,$8,$9,$10,$11,$12,$13,$14,$15,$16);");
  qs = GNUNET_PQ_eval_prepared_singleton_select (pg->conn,
                                                 "call_refresh",
                                                 params,
                                                 rs);
  GNUNET_PQ_cleanup_query_params_closures (params);
  if (0 > qs)
  {
    GNUNET_break (0);
    return qs;
  }
  if (! coin_found)
  {
    return GNUNET_DB_STATUS_SUCCESS_NO_RESULTS;
  }
  if (*found && no_noreveal_index)
  {
    GNUNET_break (0);
    return GNUNET_DB_STATUS_HARD_ERROR;
  }
  if (! balance_ok && no_coin_balance)
  {
    GNUNET_break (0);
    return GNUNET_DB_STATUS_HARD_ERROR;
  }
  return qs;
}
