/*
   This file is part of TALER
   Copyright (C) 2025 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file backenddb/pg_create_mfa_challenge.c
 * @brief Implementation of the create_mfa_challenge function for Postgres
 * @author Christian Grothoff
 */
#include "platform.h"
#include <taler/taler_error_codes.h>
#include <taler/taler_dbevents.h>
#include <taler/taler_pq_lib.h>
#include "pg_create_mfa_challenge.h"
#include "pg_helper.h"


enum GNUNET_DB_QueryStatus
TMH_PG_create_mfa_challenge (
  void *cls,
  enum TALER_MERCHANT_MFA_CriticalOperation op,
  const struct TALER_MERCHANT_MFA_BodyHash *h_body,
  const struct TALER_MERCHANT_MFA_BodySalt *salt,
  const char *code,
  struct GNUNET_TIME_Absolute expiration_date,
  struct GNUNET_TIME_Absolute retransmission_date,
  enum TALER_MERCHANT_MFA_Channel tan_channel,
  const char *required_address,
  uint64_t *challenge_id)
{
  struct PostgresClosure *pg = cls;
  struct GNUNET_TIME_Absolute now = GNUNET_TIME_absolute_get ();
  const char *op_str = TALER_MERCHANT_MFA_co_to_string (op);
  const char *channel_str = TALER_MERCHANT_MFA_channel_to_string (tan_channel);
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_auto_from_type (h_body),
    GNUNET_PQ_query_param_auto_from_type (salt),
    GNUNET_PQ_query_param_string (op_str),
    GNUNET_PQ_query_param_string (code),
    GNUNET_PQ_query_param_absolute_time (&now), /* $5 */
    GNUNET_PQ_query_param_absolute_time (&expiration_date),
    GNUNET_PQ_query_param_absolute_time (&retransmission_date),
    GNUNET_PQ_query_param_string (channel_str),
    GNUNET_PQ_query_param_string (required_address), /* $9 */
    GNUNET_PQ_query_param_end
  };
  struct GNUNET_PQ_ResultSpec rs[] = {
    GNUNET_PQ_result_spec_uint64 ("challenge_id",
                                  challenge_id),
    GNUNET_PQ_result_spec_end
  };

  PREPARE (pg,
           "create_mfa_challenge",
           "INSERT INTO tan_challenges"
           " (h_body"
           " ,salt"
           " ,op"
           " ,code"
           " ,creation_date"
           " ,expiration_date"
           " ,retransmission_date"
           " ,retry_counter" /* always set to 3 */
           " ,tan_channel"
           " ,required_address)"
           " VALUES"
           " ($1, $2, $3, $4, $5, $6, $7, 3, $8, $9)"
           " RETURNING challenge_id;");
  return GNUNET_PQ_eval_prepared_singleton_select (pg->conn,
                                                   "create_mfa_challenge",
                                                   params,
                                                   rs);
}
