﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/workmail/WorkMailRequest.h>
#include <aws/workmail/WorkMail_EXPORTS.h>

#include <utility>

namespace Aws {
namespace WorkMail {
namespace Model {

/**
 */
class PutEmailMonitoringConfigurationRequest : public WorkMailRequest {
 public:
  AWS_WORKMAIL_API PutEmailMonitoringConfigurationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "PutEmailMonitoringConfiguration"; }

  AWS_WORKMAIL_API Aws::String SerializePayload() const override;

  AWS_WORKMAIL_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ID of the organization for which the email monitoring configuration is
   * set.</p>
   */
  inline const Aws::String& GetOrganizationId() const { return m_organizationId; }
  inline bool OrganizationIdHasBeenSet() const { return m_organizationIdHasBeenSet; }
  template <typename OrganizationIdT = Aws::String>
  void SetOrganizationId(OrganizationIdT&& value) {
    m_organizationIdHasBeenSet = true;
    m_organizationId = std::forward<OrganizationIdT>(value);
  }
  template <typename OrganizationIdT = Aws::String>
  PutEmailMonitoringConfigurationRequest& WithOrganizationId(OrganizationIdT&& value) {
    SetOrganizationId(std::forward<OrganizationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the IAM Role associated with the email
   * monitoring configuration. If absent, the IAM Role Arn of
   * AWSServiceRoleForAmazonWorkMailEvents will be used.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  PutEmailMonitoringConfigurationRequest& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the CloudWatch Log group associated with
   * the email monitoring configuration.</p>
   */
  inline const Aws::String& GetLogGroupArn() const { return m_logGroupArn; }
  inline bool LogGroupArnHasBeenSet() const { return m_logGroupArnHasBeenSet; }
  template <typename LogGroupArnT = Aws::String>
  void SetLogGroupArn(LogGroupArnT&& value) {
    m_logGroupArnHasBeenSet = true;
    m_logGroupArn = std::forward<LogGroupArnT>(value);
  }
  template <typename LogGroupArnT = Aws::String>
  PutEmailMonitoringConfigurationRequest& WithLogGroupArn(LogGroupArnT&& value) {
    SetLogGroupArn(std::forward<LogGroupArnT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_organizationId;
  bool m_organizationIdHasBeenSet = false;

  Aws::String m_roleArn;
  bool m_roleArnHasBeenSet = false;

  Aws::String m_logGroupArn;
  bool m_logGroupArnHasBeenSet = false;
};

}  // namespace Model
}  // namespace WorkMail
}  // namespace Aws
