﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/clouddirectory/CloudDirectoryRequest.h>
#include <aws/clouddirectory/CloudDirectory_EXPORTS.h>
#include <aws/clouddirectory/model/BatchReadOperation.h>
#include <aws/clouddirectory/model/ConsistencyLevel.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace CloudDirectory {
namespace Model {

/**
 */
class BatchReadRequest : public CloudDirectoryRequest {
 public:
  AWS_CLOUDDIRECTORY_API BatchReadRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "BatchRead"; }

  AWS_CLOUDDIRECTORY_API Aws::String SerializePayload() const override;

  AWS_CLOUDDIRECTORY_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) that is associated with the <a>Directory</a>.
   * For more information, see <a>arns</a>.</p>
   */
  inline const Aws::String& GetDirectoryArn() const { return m_directoryArn; }
  inline bool DirectoryArnHasBeenSet() const { return m_directoryArnHasBeenSet; }
  template <typename DirectoryArnT = Aws::String>
  void SetDirectoryArn(DirectoryArnT&& value) {
    m_directoryArnHasBeenSet = true;
    m_directoryArn = std::forward<DirectoryArnT>(value);
  }
  template <typename DirectoryArnT = Aws::String>
  BatchReadRequest& WithDirectoryArn(DirectoryArnT&& value) {
    SetDirectoryArn(std::forward<DirectoryArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of operations that are part of the batch.</p>
   */
  inline const Aws::Vector<BatchReadOperation>& GetOperations() const { return m_operations; }
  inline bool OperationsHasBeenSet() const { return m_operationsHasBeenSet; }
  template <typename OperationsT = Aws::Vector<BatchReadOperation>>
  void SetOperations(OperationsT&& value) {
    m_operationsHasBeenSet = true;
    m_operations = std::forward<OperationsT>(value);
  }
  template <typename OperationsT = Aws::Vector<BatchReadOperation>>
  BatchReadRequest& WithOperations(OperationsT&& value) {
    SetOperations(std::forward<OperationsT>(value));
    return *this;
  }
  template <typename OperationsT = BatchReadOperation>
  BatchReadRequest& AddOperations(OperationsT&& value) {
    m_operationsHasBeenSet = true;
    m_operations.emplace_back(std::forward<OperationsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Represents the manner and timing in which the successful write or update of
   * an object is reflected in a subsequent read operation of that same object.</p>
   */
  inline ConsistencyLevel GetConsistencyLevel() const { return m_consistencyLevel; }
  inline bool ConsistencyLevelHasBeenSet() const { return m_consistencyLevelHasBeenSet; }
  inline void SetConsistencyLevel(ConsistencyLevel value) {
    m_consistencyLevelHasBeenSet = true;
    m_consistencyLevel = value;
  }
  inline BatchReadRequest& WithConsistencyLevel(ConsistencyLevel value) {
    SetConsistencyLevel(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_directoryArn;

  Aws::Vector<BatchReadOperation> m_operations;

  ConsistencyLevel m_consistencyLevel{ConsistencyLevel::NOT_SET};
  bool m_directoryArnHasBeenSet = false;
  bool m_operationsHasBeenSet = false;
  bool m_consistencyLevelHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudDirectory
}  // namespace Aws
