﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codeguru-security/CodeGuruSecurityRequest.h>
#include <aws/codeguru-security/CodeGuruSecurity_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace CodeGuruSecurity {
namespace Model {

/**
 */
class GetScanRequest : public CodeGuruSecurityRequest {
 public:
  AWS_CODEGURUSECURITY_API GetScanRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetScan"; }

  AWS_CODEGURUSECURITY_API Aws::String SerializePayload() const override;

  AWS_CODEGURUSECURITY_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The name of the scan you want to view details about.</p>
   */
  inline const Aws::String& GetScanName() const { return m_scanName; }
  inline bool ScanNameHasBeenSet() const { return m_scanNameHasBeenSet; }
  template <typename ScanNameT = Aws::String>
  void SetScanName(ScanNameT&& value) {
    m_scanNameHasBeenSet = true;
    m_scanName = std::forward<ScanNameT>(value);
  }
  template <typename ScanNameT = Aws::String>
  GetScanRequest& WithScanName(ScanNameT&& value) {
    SetScanName(std::forward<ScanNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>UUID that identifies the individual scan run you want to view details about.
   * You retrieve this when you call the <code>CreateScan</code> operation. Defaults
   * to the latest scan run if missing.</p>
   */
  inline const Aws::String& GetRunId() const { return m_runId; }
  inline bool RunIdHasBeenSet() const { return m_runIdHasBeenSet; }
  template <typename RunIdT = Aws::String>
  void SetRunId(RunIdT&& value) {
    m_runIdHasBeenSet = true;
    m_runId = std::forward<RunIdT>(value);
  }
  template <typename RunIdT = Aws::String>
  GetScanRequest& WithRunId(RunIdT&& value) {
    SetRunId(std::forward<RunIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_scanName;
  bool m_scanNameHasBeenSet = false;

  Aws::String m_runId;
  bool m_runIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace CodeGuruSecurity
}  // namespace Aws
