﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/evidently/CloudWatchEvidentlyRequest.h>
#include <aws/evidently/CloudWatchEvidently_EXPORTS.h>
#include <aws/evidently/model/ExperimentStopDesiredState.h>

#include <utility>

namespace Aws {
namespace CloudWatchEvidently {
namespace Model {

/**
 */
class StopExperimentRequest : public CloudWatchEvidentlyRequest {
 public:
  AWS_CLOUDWATCHEVIDENTLY_API StopExperimentRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StopExperiment"; }

  AWS_CLOUDWATCHEVIDENTLY_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>Specify whether the experiment is to be considered <code>COMPLETED</code> or
   * <code>CANCELLED</code> after it stops.</p>
   */
  inline ExperimentStopDesiredState GetDesiredState() const { return m_desiredState; }
  inline bool DesiredStateHasBeenSet() const { return m_desiredStateHasBeenSet; }
  inline void SetDesiredState(ExperimentStopDesiredState value) {
    m_desiredStateHasBeenSet = true;
    m_desiredState = value;
  }
  inline StopExperimentRequest& WithDesiredState(ExperimentStopDesiredState value) {
    SetDesiredState(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the experiment to stop.</p>
   */
  inline const Aws::String& GetExperiment() const { return m_experiment; }
  inline bool ExperimentHasBeenSet() const { return m_experimentHasBeenSet; }
  template <typename ExperimentT = Aws::String>
  void SetExperiment(ExperimentT&& value) {
    m_experimentHasBeenSet = true;
    m_experiment = std::forward<ExperimentT>(value);
  }
  template <typename ExperimentT = Aws::String>
  StopExperimentRequest& WithExperiment(ExperimentT&& value) {
    SetExperiment(std::forward<ExperimentT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name or ARN of the project that contains the experiment to stop.</p>
   */
  inline const Aws::String& GetProject() const { return m_project; }
  inline bool ProjectHasBeenSet() const { return m_projectHasBeenSet; }
  template <typename ProjectT = Aws::String>
  void SetProject(ProjectT&& value) {
    m_projectHasBeenSet = true;
    m_project = std::forward<ProjectT>(value);
  }
  template <typename ProjectT = Aws::String>
  StopExperimentRequest& WithProject(ProjectT&& value) {
    SetProject(std::forward<ProjectT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A string that describes why you are stopping the experiment.</p>
   */
  inline const Aws::String& GetReason() const { return m_reason; }
  inline bool ReasonHasBeenSet() const { return m_reasonHasBeenSet; }
  template <typename ReasonT = Aws::String>
  void SetReason(ReasonT&& value) {
    m_reasonHasBeenSet = true;
    m_reason = std::forward<ReasonT>(value);
  }
  template <typename ReasonT = Aws::String>
  StopExperimentRequest& WithReason(ReasonT&& value) {
    SetReason(std::forward<ReasonT>(value));
    return *this;
  }
  ///@}
 private:
  ExperimentStopDesiredState m_desiredState{ExperimentStopDesiredState::NOT_SET};
  bool m_desiredStateHasBeenSet = false;

  Aws::String m_experiment;
  bool m_experimentHasBeenSet = false;

  Aws::String m_project;
  bool m_projectHasBeenSet = false;

  Aws::String m_reason;
  bool m_reasonHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudWatchEvidently
}  // namespace Aws
