# frozen_string_literal: true

module GitlabQuality
  module TestTooling
    module FeatureReadiness
      module AnalyzedItems
        class AnalyzedIssue
          def initialize(issue:, token:, project:, group:, dry_run:)
            @issue = issue
            @token = token
            @project = project
            @group = group
            @dry_run = dry_run
            @analyzed_merge_requests = []
          end

          def analyze
            contributing_mrs = contributing_mrs_for_issue(issue)
            return if contributing_mrs.empty?

            @analyzed_merge_requests.concat(
              contributing_mrs
                .reject { |mr| skip_merge_request?(mr, issue) }
                .map do |merge_request|
                AnalyzedMergeRequest.new(merge_request: merge_request, token: token, project: project, group: group, dry_run: dry_run)
                                    .tap(&:analyze)
                                    .result
              end
            )
          end

          def result
            {
              issue_iid: issue.iid,
              issue_web_url: issue.web_url,
              merge_requests: analyzed_merge_requests,
              doc_mrs: doc_mrs,
              feature_spec_mrs: feature_spec_mrs,
              e2e_spec_mrs: e2e_spec_mrs,
              feature_flag_mrs: feature_flag_mrs
            }
          end

          private

          attr_reader :issue, :token, :project, :group, :dry_run, :analyzed_merge_requests

          def contributing_mrs_for_issue(issue)
            related_mrs = issue_client.related_merge_requests(iid: issue.iid)

            return [] if related_mrs.empty?

            related_mrs.select do |mr|
              mr.description&.include?("##{issue.iid}") || mr.description&.include?(issue.web_url)
            end
          end

          def skip_merge_request?(merge_request, issue)
            if (merge_request.target_project_id != issue.project_id) || (merge_request.draft == true)
              puts "Skipping: #{merge_request.web_url} #{merge_request.draft == true ? 'as it is a draft' : 'due to different project id'}"
              return true
            end

            false
          end

          def issue_client
            @issue_client ||= (dry_run ? GitlabClient::IssuesDryClient : GitlabClient::IssuesClient).new(token: token, project: project)
          end

          def doc_mrs
            analyzed_merge_requests
              .select { |mr| mr[:has_docs] }
              .map.with_index(1) { |mr, index| { "MR#{index}" => mr[:merge_request_web_url] } }
          end

          def feature_spec_mrs
            analyzed_merge_requests.select { |mr| mr[:has_feature_specs] }.map.with_index(1) { |mr, index| { "MR#{index}" => mr[:merge_request_web_url] } }
          end

          def e2e_spec_mrs
            analyzed_merge_requests.select { |mr| mr[:has_e2e_specs] }.map.with_index(1) { |mr, index| { "MR#{index}" => mr[:merge_request_web_url] } }
          end

          def feature_flag_mrs
            analyzed_merge_requests.select { |mr| mr[:added_feature_flag] }.map.with_index(1) { |mr, index| { "MR#{index}" => mr[:merge_request_web_url] } }
          end
        end
      end
    end
  end
end
