r"""
Matching covered graphs

This module implements functions and operations pertaining to matching covered
graphs.

A *matching* in a graph is a set of pairwise nonadjacent links
(nonloop edges). In other words, a matching in a graph is the edge set of an
1-regular subgraph. A matching is called a *perfect* *matching* if it the
subgraph generated by a set of matching edges spans the graph, i.e. it's the
edge set of an 1-regular spanning subgraph. A connected nontrivial graph is
called *matching* *covered* if each edge participates in some perfect matching.

{INDEX_OF_METHODS}

REFERENCES:

- This methods of this module has been adopted and inspired by the book of
  Lucchesi and Murty --- *Perfect Matchings: a theory of matching covered
  graphs* [LM2024]_.

AUTHORS:

- Janmenjaya Panda (2024-06-14): initial version

.. TODO::

    The following methods are to be incorporated in
    :class:`~MatchingCoveredGraph`:

    .. csv-table::
        :class: contentstable
        :widths: 30, 70
        :delim: |

        ``__hash__()`` | Compute a hash for ``self``, if ``self`` is immutable.
        ``_subgraph_by_deleting()`` | Return the matching covered subgraph containing the provided vertices and edges.

    **Overwritten Methods**

    .. csv-table::
        :class: contentstable
        :widths: 30, 70
        :delim: |

        ``add_clique()`` | Add a clique to the graph with the provided vertices.
        ``add_cycle()`` | Add a cycle to the graph with the provided vertices.
        ``add_path()`` | Add a path to the graph with the provided vertices.
        ``cartesian_product()`` | Return the Cartesian product of ``self`` and ``other``.
        ``clear()`` | Empties the graph of vertices and edges and removes name, associated objects, and position information.
        ``complement()`` | Return the complement of the graph.
        ``contract_edge()`` | Contract an edge from ``u`` to ``v``.
        ``contract_edges()`` | Contract edges from an iterable container.
        ``degree_constrained_subgraph()`` | Return a degree-constrained matching covered subgraph.
        ``delete_edge()`` | Delete the edge from ``u`` to ``v``.
        ``delete_edges()`` | Delete edges from an iterable container.
        ``delete_multiedge()`` | Delete all edges from ``u`` to ``v``.
        ``disjoint_union()`` | Return the disjoint union of ``self`` and ``other``.
        ``disjunctive_product()`` | Return the disjunctive product of ``self`` and ``other``.
        ``is_block_graph()`` | Check whether the matching covered graph is a block graph.
        ``is_cograph()`` | Check whether the matching covered graph is cograph.
        ``is_forest()`` | Check if the matching covered graph is a forest, i.e. a disjoint union of trees.
        ``is_matching_covered()`` | Check if the graph is matching covered.
        ``is_path()`` | Check whether the graph is a path.
        ``is_subgraph()`` | Check whether the matching covered graph is a subgraph of ``other``.
        ``is_tree()`` | Check whether the matching covered graph is a tree.
        ``join()`` | Return the join of ``self`` and ``other``.
        ``lexicographic_product()`` | Return the lexicographic product of ``self`` and ``other``.
        ``load_afile()`` | Load the matching covered graph specified in the given file into the current object.
        ``merge_vertices()`` | Merge vertices.
        ``minor()`` | Return the vertices of a minor isomorphic to `H` in the current graph.
        ``random_subgraph()`` | Return a random matching covered subgraph containing each vertex with probability ``p``.
        ``save_afile()`` | Save the graph to file in alist format.
        ``strong_product()`` | Return the strong product of ``self`` and ``other``.
        ``subgraph()`` | Return the matching covered subgraph containing the given vertices and edges.
        ``subgraph_search()`` | Return a copy of (matching covered) ``G`` in ``self``.
        ``subgraph_search_count()`` | Return the number of labelled occurrences of (matching covered) ``G`` in ``self``.
        ``subgraph_search_iterator()`` | Return an iterator over the labelled copies of (matching covered) ``G`` in ``self``.
        ``tensor_product()`` | Return the tensor product of ``self`` and ``other``.
        ``to_undirected()`` | Return an undirected Graph instance of the matching covered graph.
        ``topological_minor()`` | Return a topological `H`-minor from ``self`` if one exists.
        ``transitive_closure()`` | Return the transitive closure of the matching covered graph.
        ``transitive_reduction()`` | Return a transitive reduction of the matching covered graph.
        ``union()`` | Return the union of ``self`` and ``other``.

    **Bricks, braces and tight cut decomposition**

    .. csv-table::
        :class: contentstable
        :widths: 30, 70
        :delim: |

        ``bricks_and_braces()`` | Return the list of (underlying simple graph of) the bricks and braces of the (matching covered) graph.
        ``number_of_braces()`` | Return the number of braces.
        ``number_of_bricks()`` | Return the number of bricks.
        ``number_of_petersen_bricks()`` | Return the number of Petersen bricks.
        ``tight_cut_decomposition()`` | Return a maximal set of laminar nontrivial tight cuts and a corresponding vertex set partition.

    **Removability and ear decomposition**

    .. csv-table::
        :class: contentstable
        :widths: 30, 70
        :delim: |

        ``add_ear()`` | Add an ear to the graph with the provided end vertices number of internal vertices.
        ``efficient_ear_decomposition()`` | Return a matching covered ear decomposition computed at the fastest possible time.
        ``is_removable_double_ear()`` | Check whether the pair of ears form a removable double ear.
        ``is_removable_doubleton()`` | Check whether the pair of edges constitute a removable doubleton.
        ``is_removable_ear()`` | Check whether the ear is removable.
        ``is_removable_edge()`` | Check whether the edge is removable.
        ``optimal_ear_decomposition()`` | Return an optimal ear decomposition.
        ``removable_double_ears()`` | Return a list of removable double ears.
        ``removable_doubletons()`` | Return a list of removable doubletons.
        ``removable_ears()`` | Return a list of removable ears.
        ``removable_edges()`` | Return a :class:`~EdgesView` of removable edges.
        ``retract()`` | Compute the retract of the (matching covered) graph.

    **Generating bricks and braces**

    .. csv-table::
        :class: contentstable
        :widths: 30, 70
        :delim: |

        ``brace_generation_sequence()`` | Return a McCuaig brace generation sequence of the (provided) brace.
        ``brick_generation_sequence()`` | Return a Norine-Thomas brick generation sequence of the (provided) brick.
        ``is_mccuaig_brace()`` | Check if the brace is a McCuaig brace.
        ``is_norine_thomas_brick()`` | Check if the brick is a Norine-Thomas brick.


Methods
-------
"""

# ****************************************************************************
#         Copyright (C) 2024 Janmenjaya Panda <janmenjaya.panda.22@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 2 of the License, or
# (at your option) any later version.
#                  https://www.gnu.org/licenses/
# ****************************************************************************
from sage.graphs.graph import Graph
from sage.misc.rest_index_of_methods import doc_index, gen_thematic_rest_table_index


class MatchingCoveredGraph(Graph):
    r"""
    Matching covered graph

    INPUT:

    - ``data`` -- can be any of the following:

      - Empty or ``None`` (throws a :exc:`ValueError` as the graph must be
        nontrival).

      - An arbitrary graph.

    - ``matching`` -- (default: ``None``); a perfect matching of the
      graph, that can be given using any valid input format of
      :class:`~sage.graphs.graph.Graph`.

      If set to ``None``, a matching is computed using the other parameters.

    - ``algorithm`` -- string (default: ``'Edmonds'``); the algorithm to be
      used to compute a maximum matching of the graph among

      - ``'Edmonds'`` selects Edmonds' algorithm as implemented in NetworkX,

      - ``'LP'`` uses a Linear Program formulation of the matching problem.

    - ``solver`` -- string (default: ``None``); specify a Mixed Integer
      Linear Programming (MILP) solver to be used. If set to ``None``, the
      default one is used. For more information on MILP solvers and which
      default solver is used, see the method :meth:`solve
      <sage.numerical.mip.MixedIntegerLinearProgram.solve>` of the class
      :class:`MixedIntegerLinearProgram
      <sage.numerical.mip.MixedIntegerLinearProgram>`.

    - ``verbose`` -- integer (default: ``0``); sets the level of verbosity:
      set to 0 by default, which means quiet (only useful when ``algorithm
      == 'LP'``).

    - ``integrality_tolerance`` -- float; parameter for use with MILP
      solvers over an inexact base ring; see
      :meth:`MixedIntegerLinearProgram.get_values`.

    OUTPUT:

    - An object of the class :class:`~MatchingCoveredGraph` if the input is
      valid and the graph is matching covered, or otherwise an error is thrown.

    .. NOTE::

        All remaining arguments are passed to the ``Graph`` constructor

    EXAMPLES:

    Generating an object of the class ``MatchingCoveredGraph`` from the
    provided instance of ``Graph`` without providing any other information::

        sage: G = MatchingCoveredGraph(graphs.PetersenGraph())
        sage: G
        Matching covered petersen graph: graph on 10 vertices
        sage: sorted(G.get_matching())
        [(0, 5, None), (1, 6, None), (2, 7, None), (3, 8, None), (4, 9, None)]

        sage: G = graphs.StaircaseGraph(4)
        sage: H = MatchingCoveredGraph(G)
        sage: H
        Matching covered staircase graph: graph on 8 vertices
        sage: H == G
        True
        sage: sorted(H.get_matching())
        [(0, 1, None), (2, 7, None), (3, 6, None), (4, 5, None)]

        sage: G = Graph({0: [1, 2, 3, 4], 1: [2, 5],
        ....:            2: [5], 3: [4, 5], 4: [5]})
        sage: H = MatchingCoveredGraph(G)
        sage: H
        Matching covered graph on 6 vertices
        sage: H == G
        True
        sage: sorted(H.get_matching())
        [(0, 4, None), (1, 2, None), (3, 5, None)]

        sage: # needs networkx
        sage: import networkx
        sage: G = Graph(networkx.complete_bipartite_graph(12, 12))
        sage: H = MatchingCoveredGraph(G)
        sage: H
        Matching covered graph on 24 vertices
        sage: H == G
        True
        sage: sorted(H.get_matching())
        [(0, 15, None), (1, 14, None), (2, 13, None), (3, 12, None),
         (4, 23, None), (5, 22, None), (6, 21, None), (7, 20, None),
         (8, 19, None), (9, 18, None), (10, 17, None), (11, 16, None)]

        sage: G = Graph('E|fG', sparse=True)
        sage: H = MatchingCoveredGraph(G)
        sage: H
        Matching covered graph on 6 vertices
        sage: H == G
        True
        sage: sorted(H.get_matching())
        [(0, 5, None), (1, 2, None), (3, 4, None)]

        sage: # needs sage.modules
        sage: M = Matrix([(0,1,0,0,1,1,0,0,0,0),
        ....:             (1,0,1,0,0,0,1,0,0,0),
        ....:             (0,1,0,1,0,0,0,1,0,0),
        ....:             (0,0,1,0,1,0,0,0,1,0),
        ....:             (1,0,0,1,0,0,0,0,0,1),
        ....:             (1,0,0,0,0,0,0,1,1,0),
        ....:             (0,1,0,0,0,0,0,0,1,1),
        ....:             (0,0,1,0,0,1,0,0,0,1),
        ....:             (0,0,0,1,0,1,1,0,0,0),
        ....:             (0,0,0,0,1,0,1,1,0,0)])
        sage: M
        [0 1 0 0 1 1 0 0 0 0]
        [1 0 1 0 0 0 1 0 0 0]
        [0 1 0 1 0 0 0 1 0 0]
        [0 0 1 0 1 0 0 0 1 0]
        [1 0 0 1 0 0 0 0 0 1]
        [1 0 0 0 0 0 0 1 1 0]
        [0 1 0 0 0 0 0 0 1 1]
        [0 0 1 0 0 1 0 0 0 1]
        [0 0 0 1 0 1 1 0 0 0]
        [0 0 0 0 1 0 1 1 0 0]
        sage: G = Graph(M)
        sage: H = MatchingCoveredGraph(G)
        sage: H == G
        True
        sage: sorted(H.get_matching())
        [(0, 5, None), (1, 6, None), (2, 7, None), (3, 8, None), (4, 9, None)]

        sage: # needs sage.modules
        sage: M = Matrix([(-1, 0, 0, 0, 1, 0, 0, 0, 0, 0,-1, 0, 0, 0, 0),
        ....:             ( 1,-1, 0, 0, 0, 0, 0, 0, 0, 0, 0,-1, 0, 0, 0),
        ....:             ( 0, 1,-1, 0, 0, 0, 0, 0, 0, 0, 0, 0,-1, 0, 0),
        ....:             ( 0, 0, 1,-1, 0, 0, 0, 0, 0, 0, 0, 0, 0,-1, 0),
        ....:             ( 0, 0, 0, 1,-1, 0, 0, 0, 0, 0, 0, 0, 0, 0,-1),
        ....:             ( 0, 0, 0, 0, 0,-1, 0, 0, 0, 1, 1, 0, 0, 0, 0),
        ....:             ( 0, 0, 0, 0, 0, 0, 0, 1,-1, 0, 0, 1, 0, 0, 0),
        ....:             ( 0, 0, 0, 0, 0, 1,-1, 0, 0, 0, 0, 0, 1, 0, 0),
        ....:             ( 0, 0, 0, 0, 0, 0, 0, 0, 1,-1, 0, 0, 0, 1, 0),
        ....:             ( 0, 0, 0, 0, 0, 0, 1,-1, 0, 0, 0, 0, 0, 0, 1)])
        sage: M
        [-1  0  0  0  1  0  0  0  0  0 -1  0  0  0  0]
        [ 1 -1  0  0  0  0  0  0  0  0  0 -1  0  0  0]
        [ 0  1 -1  0  0  0  0  0  0  0  0  0 -1  0  0]
        [ 0  0  1 -1  0  0  0  0  0  0  0  0  0 -1  0]
        [ 0  0  0  1 -1  0  0  0  0  0  0  0  0  0 -1]
        [ 0  0  0  0  0 -1  0  0  0  1  1  0  0  0  0]
        [ 0  0  0  0  0  0  0  1 -1  0  0  1  0  0  0]
        [ 0  0  0  0  0  1 -1  0  0  0  0  0  1  0  0]
        [ 0  0  0  0  0  0  0  0  1 -1  0  0  0  1  0]
        [ 0  0  0  0  0  0  1 -1  0  0  0  0  0  0  1]
        sage: G = Graph(M)
        sage: H = MatchingCoveredGraph(G)
        sage: H == G
        True
        sage: sorted(H.get_matching())
        [(0, 5, None), (1, 6, None), (2, 7, None), (3, 8, None), (4, 9, None)]

        sage: G = Graph([(0, 1), (0, 3), (0, 4), (1, 2), (1, 5), (2, 3),
        ....:            (2, 6), (3, 7), (4, 5), (4, 7), (5, 6), (6, 7)])
        sage: H = MatchingCoveredGraph(G)
        sage: H == G
        True
        sage: sorted(H.get_matching())
        [(0, 4, None), (1, 5, None), (2, 6, None), (3, 7, None)]

        sage: # optional - python_igraph
        sage: import igraph
        sage: G = Graph(igraph.Graph([(0, 1), (0, 3), (1, 2), (2, 3)]))
        sage: H = MatchingCoveredGraph(G)
        sage: H
        Matching covered graph on 4 vertices
        sage: sorted(H.get_matching())
        [(0, 3, {}), (1, 2, {})]

    One may specify a perfect matching::

        sage: P = graphs.PetersenGraph()
        sage: M = P.matching()
        sage: sorted(M)
        [(0, 5, None), (1, 6, None), (2, 7, None), (3, 8, None), (4, 9, None)]
        sage: G = MatchingCoveredGraph(P, matching=M)
        sage: G
        Matching covered petersen graph: graph on 10 vertices
        sage: P == G
        True
        sage: sorted(G.get_matching())
        [(0, 5, None), (1, 6, None), (2, 7, None), (3, 8, None), (4, 9, None)]
        sage: sorted(G.get_matching()) == sorted(M)
        True

        sage: G = graphs.TruncatedBiwheelGraph(14)
        sage: M = G.matching()
        sage: sorted(M)
        [(0, 27, None), (1, 26, None), (2, 3, None), (4, 5, None),
         (6, 7, None), (8, 9, None), (10, 11, None), (12, 13, None),
         (14, 15, None), (16, 17, None), (18, 19, None), (20, 21, None),
         (22, 23, None), (24, 25, None)]
        sage: H = MatchingCoveredGraph(G, M)
        sage: H
        Matching covered truncated biwheel graph: graph on 28 vertices
        sage: H == G
        True
        sage: sorted(H.get_matching()) == sorted(M)
        True

    One may specify some keyword arguments::

        sage: G = Graph([(0, 1, 5)], {'weighted': True})
        sage: kwds = {
        ....:   'loops': False,
        ....:   'multiedges': True,
        ....:   'pos': {0: (0, 0), 1: (1, 1)}
        ....: }
        sage: H = MatchingCoveredGraph(G, **kwds)
        sage: H
        Matching covered multi-graph on 2 vertices
        sage: H.add_edge(0, 1)
        sage: H.edges()
        [(0, 1, None), (0, 1, 5)]

    TESTS:

    An empty graph is not matching covered::

        sage: G = Graph()
        sage: H = MatchingCoveredGraph(G)
        Traceback (most recent call last):
        ...
        ValueError: the graph is trivial
        sage: G = MatchingCoveredGraph()
        Traceback (most recent call last):
        ...
        ValueError: the graph is trivial

    Providing with a graph that is not connected::

        sage: G = graphs.CycleGraph(4)
        sage: G += graphs.CycleGraph(6)
        sage: G.number_of_connected_components()
        2
        sage: H = MatchingCoveredGraph(G)
        Traceback (most recent call last):
        ...
        ValueError: the graph is not connected

    Make sure that self-loops are not allowed for a matching covered graph::

        sage: P = graphs.PetersenGraph()
        sage: kwds = {'loops': True}
        sage: G = MatchingCoveredGraph(P, **kwds)
        Traceback (most recent call last):
        ...
        ValueError: loops are not allowed in matching covered graphs
        sage: G = MatchingCoveredGraph(P)
        sage: G.allows_loops()
        False
        sage: G.allow_loops(True)
        Traceback (most recent call last):
        ...
        ValueError: loops are not allowed in matching covered graphs
        sage: G.add_edge(0, 0)
        Traceback (most recent call last):
        ...
        ValueError: loops are not allowed in matching covered graphs
        sage: H = MatchingCoveredGraph(P, loops=True)
        Traceback (most recent call last):
        ...
        ValueError: loops are not allowed in matching covered graphs

    Make sure that multiple edges are allowed for a matching covered graph (by
    default it is off and can be modified to be allowed)::

        sage: P = graphs.PetersenGraph()
        sage: G = MatchingCoveredGraph(P)
        sage: G
        Matching covered petersen graph: graph on 10 vertices
        sage: G.allows_multiple_edges()
        False
        sage: G.size()
        15
        sage: G.allow_multiple_edges(True)
        sage: G.allows_multiple_edges()
        True
        sage: G.add_edge(next(P.edge_iterator()))
        sage: G.size()
        16
        sage: G
        Matching covered petersen graph: multi-graph on 10 vertices
        sage: H = MatchingCoveredGraph(P, multiedges=True)
        sage: H.allows_multiple_edges()
        True
        sage: H.add_edge(next(P.edge_iterator()))
        sage: H.size()
        16
        sage: H
        Matching covered petersen graph: multi-graph on 10 vertices

    Providing with a connected nontrivial graph free of self-loops that is
    not matching covered::

        sage: G = graphs.CompleteGraph(11)
        sage: H = MatchingCoveredGraph(G)
        Traceback (most recent call last):
        ...
        ValueError: input graph is not matching covered
        sage: G = Graph({0: [1, 6, 11], 1: [2, 4], 2: [3, 5], 3: [4, 5],
        ....:            4: [5], 6: [7, 9], 7: [8, 10], 8: [9, 10], 9: [10],
        ....:            11: [12, 14], 12: [13, 15], 13: [14, 15], 14: [15]})
        sage: H = MatchingCoveredGraph(G)
        Traceback (most recent call last):
        ...
        ValueError: input graph is not matching covered

        sage: # needs networkx
        sage: import networkx
        sage: G = Graph(networkx.complete_bipartite_graph(2, 12))
        sage: H = MatchingCoveredGraph(G)
        Traceback (most recent call last):
        ...
        ValueError: input graph is not matching covered
        sage: G = Graph('F~~~w')
        sage: H = MatchingCoveredGraph(G)
        Traceback (most recent call last):
        ...
        ValueError: input graph is not matching covered

        sage: # needs sage.modules
        sage: M = Matrix([(0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0),
        ....:             (1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0),
        ....:             (0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0),
        ....:             (0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0),
        ....:             (0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0),
        ....:             (0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0),
        ....:             (1, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0),
        ....:             (0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0),
        ....:             (0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 0),
        ....:             (0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0, 0, 0),
        ....:             (0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0),
        ....:             (1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0),
        ....:             (0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 1),
        ....:             (0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 1),
        ....:             (0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 1),
        ....:             (0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0)])
        sage: M
        [0 1 0 0 0 0 1 0 0 0 0 1 0 0 0 0]
        [1 0 1 0 1 0 0 0 0 0 0 0 0 0 0 0]
        [0 1 0 1 0 1 0 0 0 0 0 0 0 0 0 0]
        [0 0 1 0 1 1 0 0 0 0 0 0 0 0 0 0]
        [0 1 0 1 0 1 0 0 0 0 0 0 0 0 0 0]
        [0 0 1 1 1 0 0 0 0 0 0 0 0 0 0 0]
        [1 0 0 0 0 0 0 1 0 1 0 0 0 0 0 0]
        [0 0 0 0 0 0 1 0 1 0 1 0 0 0 0 0]
        [0 0 0 0 0 0 0 1 0 1 1 0 0 0 0 0]
        [0 0 0 0 0 0 1 0 1 0 1 0 0 0 0 0]
        [0 0 0 0 0 0 0 1 1 1 0 0 0 0 0 0]
        [1 0 0 0 0 0 0 0 0 0 0 0 1 0 1 0]
        [0 0 0 0 0 0 0 0 0 0 0 1 0 1 0 1]
        [0 0 0 0 0 0 0 0 0 0 0 0 1 0 1 1]
        [0 0 0 0 0 0 0 0 0 0 0 1 0 1 0 1]
        [0 0 0 0 0 0 0 0 0 0 0 0 1 1 1 0]
        sage: G = Graph(M)
        sage: H = MatchingCoveredGraph(G)
        Traceback (most recent call last):
        ...
        ValueError: input graph is not matching covered

        sage: # needs sage.modules
        sage: M = Matrix([(1, 1, 0, 0, 0, 0),
        ....:             (0, 0, 1, 1, 0, 0),
        ....:             (0, 0, 1, 0, 1, 0),
        ....:             (1, 0, 0, 0, 0, 1),
        ....:             (0, 1, 0, 1, 1, 1)])
        sage: G = Graph(M)
        sage: H = MatchingCoveredGraph(G)
        Traceback (most recent call last):
        ...
        ValueError: input graph is not matching covered
        sage: G = Graph([(11, 12), (11, 14), (0, 1), (0, 11), (0, 6), (1, 2),
        ....:            (1, 4), (2, 3), (2, 5), (3, 4), (3, 5), (4, 5),
        ....:            (6, 7), (6, 9), (7, 8), (7, 10), (8, 9), (8, 10),
        ....:            (9, 10), (12, 13), (12, 15), (13, 14), (13, 15),
        ....:            (14, 15)])
        sage: H = MatchingCoveredGraph(G)
        Traceback (most recent call last):
        ...
        ValueError: input graph is not matching covered

        sage: # optional - python_igraph
        sage: import igraph
        sage: G = Graph(igraph.Graph([(0, 1), (0, 2), (0, 3), (1, 2), (2, 3)]))
        sage: H = MatchingCoveredGraph(G)
        Traceback (most recent call last):
        ...
        ValueError: input graph is not matching covered

    Providing with a wrong matching::

        sage: P = graphs.PetersenGraph()
        sage: M = str('0')
        sage: H = MatchingCoveredGraph(P, matching=M)
        Traceback (most recent call last):
        ...
        RuntimeError: the string seems corrupt: valid characters are
        ?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\]^_`abcdefghijklmnopqrstuvwxyz{|}~
        sage: N = str('graph')
        sage: J = MatchingCoveredGraph(P, matching=N)
        Traceback (most recent call last):
        ...
        RuntimeError: the string (graph) seems corrupt: for n = 40,
        the string is too short

        sage: G = graphs.CompleteGraph(6)
        sage: M = Graph(G.matching())
        sage: M.add_edges([(0, 1), (0, 2)])
        sage: H = MatchingCoveredGraph(G, matching=M)
        Traceback (most recent call last):
        ...
        ValueError: the input is not a matching
        sage: N = Graph(G.matching())
        sage: N.add_edge(6, 7)
        sage: H = MatchingCoveredGraph(G, matching=N)
        Traceback (most recent call last):
        ...
        ValueError: the input is not a matching of the graph
        sage: J = Graph()
        sage: J.add_edges([(0, 1), (2, 3)])
        sage: H = MatchingCoveredGraph(G, matching=J)
        Traceback (most recent call last):
        ...
        ValueError: the input is not a perfect matching of the graph

    Note that data shall be one of empty or ``None`` or an instance of
    ``Graph`` or an instance of ``MatchingCoveredGraph``. Otherwise a
    :exc:`ValueError` is returned::

        sage: D = digraphs.Complete(10)
        sage: D
        Complete digraph: Digraph on 10 vertices
        sage: G = MatchingCoveredGraph(D)
        Traceback (most recent call last):
        ...
        TypeError: input data is of unknown type
    """

    def __init__(self, data=None, matching=None, algorithm='Edmonds',
                 solver=None, verbose=0, integrality_tolerance=0.001,
                 *args, **kwds):
        r"""
        Create a matching covered graph, that is a connected nontrivial graph
        wherein each edge participates in some perfect matching.

        See documentation ``MatchingCoveredGraph?`` for detailed information.
        """
        success = False

        if not kwds:
            kwds = {'loops': False}
        else:
            if 'loops' in kwds and kwds['loops']:
                raise ValueError('loops are not allowed in '
                                 'matching covered graphs')
            kwds['loops'] = False

        if data is None:
            raise ValueError('the graph is trivial')

        elif isinstance(data, MatchingCoveredGraph):
            Graph.__init__(self, data, *args, **kwds)
            success = True

        elif isinstance(data, Graph):
            try:
                self._upgrade_from_graph(data=data, matching=matching,
                                         algorithm=algorithm,
                                         solver=solver, verbose=verbose,
                                         integrality_tolerance=integrality_tolerance,
                                         *args, **kwds)
                success = True

            except Exception as exception:
                raise exception

        if success:
            if matching:
                # The input matching is a valid perfect matching of the graph
                self._matching = matching

            else:
                self._matching = Graph(self).matching()

        else:
            raise TypeError('input data is of unknown type')

    def __repr__(self):
        r"""
        Return a short string representation of the (matching covered) graph.

        EXAMPLES:

        If the string representation of the (matching covered) graph does not
        contain the term 'matching covered', it's used as the prefix::

            sage: G = graphs.CompleteGraph(10)
            sage: H = MatchingCoveredGraph(G)
            sage: H
            Matching covered complete graph: graph on 10 vertices

            sage: G = graphs.HexahedralGraph()
            sage: H = MatchingCoveredGraph(BipartiteGraph(G))
            sage: H  # An object of the class MatchingCoveredGraph
            Matching covered hexahedron: graph on 8 vertices

        In case the string representation of the (matching covered) graph
        contains the term 'matching covered', the representation remains as it
        is::

            sage: G = graphs.CompleteGraph(10)
            sage: H = MatchingCoveredGraph(G)
            sage: H
            Matching covered complete graph: graph on 10 vertices
            sage: J = MatchingCoveredGraph(H)
            sage: J
            Matching covered complete graph: graph on 10 vertices
            sage: G = graphs.HexahedralGraph()
            sage: H = BipartiteGraph(MatchingCoveredGraph(G))
            sage: H  # An object of the class BipartiteGraph
            Bipartite hexahedron: graph on 8 vertices
            sage: J = MatchingCoveredGraph(H)
            sage: J  # An object of the class MatchingCoveredGraph
            Matching covered hexahedron: graph on 8 vertices
        """
        s = Graph._repr_(self).lower()
        if "matching covered" in s:
            return s.capitalize()
        return "".join(["Matching covered ", s])

    def _subgraph_by_adding(self, vertices=None, edges=None, edge_property=None, immutable=None):
        r"""
        Return the matching covered subgraph containing the given vertices and edges.

        The edges also satisfy the edge_property, if it is not None. The
        subgraph is created by creating a new empty graph and adding the
        necessary vertices, edges, and other properties.

        .. NOTE::

            This method overwrites the
            :meth:`~sage.graphs.generic_graph.GenericGraph._subgraph_by_adding`
            method to ensure that resultant subgraph is also matching covered.

        INPUT:

        - ``vertices`` -- (default: ``None``) an iterable container of
          vertices, e.g. a list, set, graph, file or numeric array. If not
          passed (i.e., ``None``), defaults to the entire graph.

        - ``edges`` -- a single edge or an iterable container of edges (e.g., a
          list, set, file, numeric array, etc.). By default (``edges=None``),
          all edges are assumed and the returned graph is an induced
          subgraph. In the case of multiple edges, specifying an edge as `(u,v)`
          means to keep all edges `(u,v)`, regardless of the label.

        - ``edge_property`` -- function (default: ``None``); a function that
          inputs an edge and outputs a boolean value, i.e., a edge ``e`` in
          ``edges`` is kept if ``edge_property(e) == True``

        - ``immutable`` -- boolean (default: ``None``); whether to create a
          mutable/immutable subgraph. ``immutable=None`` (default) means that
          the graph and its subgraph will behave the same way.

        OUTPUT:

        - An instance of :class:`~MatchingCoveredGraph` is returned if the
          subgraph obtained is matching covered, otherwise a :exc:`ValueError`
          is thrown.

        EXAMPLES:

        Ladder graphs are matching covered subgraphs of a staircase graph,
        which is also matching covered::

            sage: G = MatchingCoveredGraph(graphs.StaircaseGraph(4))
            sage: H = G._subgraph_by_adding(vertices=[0..5])
            sage: H.order(), H.size()
            (6, 7)
            sage: H
            Matching covered subgraph of (staircase graph): graph on 6 vertices
            sage: H.is_isomorphic(graphs.LadderGraph(3))
            True

        Cycle graphs are matching covered subgraphs of a biwheel graph, which
        is also matching covered::

            sage: G = MatchingCoveredGraph(graphs.BiwheelGraph(5))
            sage: H = G._subgraph_by_adding(vertices=[0..7],
            ....:     edges=[(u, (u+1) % 8) for u in range(8)])
            sage: H.order(), H.size()
            (8, 8)
            sage: H
            Matching covered subgraph of (biwheel graph): graph on 8 vertices
            sage: H.is_isomorphic(graphs.CycleGraph(8))
            True

        One may pass no value for any of the input arguments; in such a case,
        the whole matching covered graph will be returned::

            sage: T = graphs.TwinplexGraph()
            sage: G = MatchingCoveredGraph(T)
            sage: J = G._subgraph_by_adding()
            sage: G == J
            True

        One may use the ``edge_property`` argument::

            sage: G = Graph(multiedges=True)
            sage: G.add_edges([
            ....:     (0, 1, 'label'), (0, 2), (0, 3), (0, 4),
            ....:     (0, 5), (1, 2, 'label'), (1, 2), (1, 5),
            ....:     (2, 5), (3, 4), (3, 5), (4, 5)
            ....: ])
            sage: H = MatchingCoveredGraph(G)
            sage: J = H._subgraph_by_adding(vertices=[0, 1, 2, 5], edge_property=
            ....:     (lambda edge:
            ....:             (edge[0] in [1, 2]) != (edge[1] in [1, 2]))
            ....: )
            sage: J.order(), J.size()
            (4, 4)
            sage: J
            Matching covered subgraph of (): multi-graph on 4 vertices
            sage: J.is_isomorphic(graphs.CompleteBipartiteGraph(2, 2))
            True

        We may specify the subgraph to be immutable::

            sage: M = graphs.MoebiusLadderGraph(4)
            sage: G = MatchingCoveredGraph(M)
            sage: H = G._subgraph_by_adding(edge_property=
            ....:     (lambda edge: abs(edge[0] - edge[1]) != 4),
            ....:     immutable=True)
            sage: H.order(), H.size()
            (8, 8)
            sage: H
            Matching covered subgraph of (moebius ladder graph): graph on 8 vertices
            sage: H.is_isomorphic(graphs.CycleGraph(8))
            True
            sage: H.is_immutable()
            True
            sage: C = graphs.CubeplexGraph()
            sage: D = MatchingCoveredGraph(C)
            sage: I = D._subgraph_by_adding(immutable=True)
            sage: (I == D) and (I.is_immutable())
            True
            sage: J = D._subgraph_by_adding(vertices=D.vertices(), immutable=True)
            sage: (J == D) and (J.is_immutable())
            True

        An error is thrown if the subgraph is not matching covered::

            sage: P = graphs.PetersenGraph()
            sage: G = MatchingCoveredGraph(P)
            sage: H = G._subgraph_by_adding(vertices=[])
            Traceback (most recent call last):
            ...
            ValueError: the graph is trivial
            sage: H = G._subgraph_by_adding(edge_property=
            ....:     (lambda edge: edge[0] == 0)
            ....: )
            Traceback (most recent call last):
            ...
            ValueError: the graph is not connected
            sage: H = G._subgraph_by_adding(vertices=[1, 2, 3])
            Traceback (most recent call last):
            ...
            ValueError: input graph is not matching covered
        """
        if immutable is None:
            immutable = self.is_immutable()

        if edges is None and edge_property is None:
            if vertices is None:
                G = self.copy()
                G.name('Matching covered subgraph of ({})'.format(self.name()))
                if immutable:
                    G = G.copy(immutable=True)

                return G

            else:
                # Check if all existent vertices are there
                all_existent_vertices = True
                for vertex in self:
                    if vertex not in vertices:
                        all_existent_vertices = False
                        break

                if all_existent_vertices:
                    G = self.copy()
                    G.name('Matching covered subgraph of ({})'.format(self.name()))
                    if immutable:
                        G = G.copy(immutable=True)

                    return G

        G = Graph(self, weighted=self._weighted, loops=self.allows_loops(),
                  multiedges=self.allows_multiple_edges())

        H = G._subgraph_by_adding(vertices=vertices, edges=edges,
                                  edge_property=edge_property,
                                  immutable=False)

        try:
            H = MatchingCoveredGraph(H)
            H.name('Matching covered subgraph of ({})'.format(self.name()))
            if immutable:
                H = H.copy(immutable=True)

            return H

        except Exception as exception:
            raise exception

    def _upgrade_from_graph(self, data=None, matching=None, algorithm='Edmonds',
                            solver=None, verbose=0, integrality_tolerance=0.001,
                            *args, **kwds):
        r"""
        Upgrade the given graph to a matching covered graph if eligible.

        See documentation ``MatchingCoveredGraph?`` for detailed information.
        """
        try:
            check = Graph.is_matching_covered(G=data, matching=matching,
                                              algorithm=algorithm,
                                              coNP_certificate=False,
                                              solver=solver, verbose=verbose,
                                              integrality_tolerance=integrality_tolerance)

            if check:
                Graph.__init__(self, data, *args, **kwds)
            else:
                raise ValueError("input graph is not matching covered")

        except Exception as exception:
            raise exception

    @doc_index('Overwritten methods')
    def add_edge(self, u, v=None, label=None):
        r"""
        Add an edge from vertex ``u`` to vertex ``v``.

        .. NOTE::

            This method overwrites the
            :meth:`~sage.graphs.generic_graph.GenericGraph.add_edge` method
            to ensure that resultant graph is also matching covered.

        INPUT:

        The following forms are all accepted:

        - G.add_edge(1, 2)
        - G.add_edge((1, 2))
        - G.add_edges([(1, 2)])
        - G.add_edge(1, 2, 'label')
        - G.add_edge((1, 2, 'label'))
        - G.add_edges([(1, 2, 'label')])

        OUTPUT:

        - If an edge is provided with a valid format, but addition of the edge
          leaves the resulting graph not being matching covered, a
          :exc:`ValueError` is returned without any alteration to the existing
          matching covered graph. If the addition of the edge preserves the
          property of matching covered, then the graph is updated and nothing
          is returned.

        - If the edge is provided in an invalid format, a :exc:`ValueError`
          is returned.

        WARNING:

        The following intuitive input results in nonintuitive output,
        even though the resulting graph behind the intuition might be matching
        covered::

            sage: P = graphs.WheelGraph(6)
            sage: G = MatchingCoveredGraph(P)
            sage: G.add_edge((1, 4), 'label')
            Traceback (most recent call last):
            ...
            ValueError: the graph obtained after the addition of edge
            (((1, 4), 'label', None)) is not matching covered
            sage: G.edges(sort=False)
            [(0, 1, None), (0, 2, None), (0, 3, None), (0, 4, None),
             (0, 5, None), (1, 2, None), (1, 5, None), (2, 3, None),
             (3, 4, None), (4, 5, None)]

        The key word ``label`` must be used::

            sage: W = graphs.WheelGraph(6)
            sage: G = MatchingCoveredGraph(W)
            sage: G.add_edge((1, 4), label='label')
            sage: G.edges(sort=False)  # No alteration to the existing graph
            [(0, 1, None), (0, 2, None), (0, 3, None), (0, 4, None),
             (0, 5, None), (1, 2, None), (1, 4, 'label'), (1, 5, None),
             (2, 3, None), (3, 4, None), (4, 5, None)]

        An expression, analogous to the syntax mentioned above may be used::

            sage: S = graphs.StaircaseGraph(4)
            sage: G = MatchingCoveredGraph(S)
            sage: G.add_edge(0, 5)
            sage: G.edges(sort=False)
            [(0, 1, None), (0, 3, None), (0, 5, None), (0, 6, None),
             (1, 2, None), (1, 4, None), (2, 5, None), (2, 7, None),
             (3, 4, None), (3, 6, None), (4, 5, None), (5, 7, None),
             (6, 7, None)]
            sage: G.add_edge((2, 3))
            sage: G.edges(sort=False)
            [(0, 1, None), (0, 3, None), (0, 5, None), (0, 6, None),
             (1, 2, None), (1, 4, None), (2, 3, None), (2, 5, None),
             (2, 7, None), (3, 4, None), (3, 6, None), (4, 5, None),
             (5, 7, None), (6, 7, None)]
            sage: G.add_edges([(0, 4)])
            sage: G.edges(sort=False)
            [(0, 1, None), (0, 3, None), (0, 4, None), (0, 5, None),
             (0, 6, None), (1, 2, None), (1, 4, None), (2, 3, None),
             (2, 5, None), (2, 7, None), (3, 4, None), (3, 6, None),
             (4, 5, None), (5, 7, None), (6, 7, None)]
            sage: G.add_edge(2, 4, 'label')
            sage: G.edges(sort=False)
            [(0, 1, None), (0, 3, None), (0, 4, None), (0, 5, None),
             (0, 6, None), (1, 2, None), (1, 4, None), (2, 3, None),
             (2, 4, 'label'), (2, 5, None), (2, 7, None), (3, 4, None),
             (3, 6, None), (4, 5, None), (5, 7, None), (6, 7, None)]
            sage: G.add_edge((4, 6, 'label'))
            sage: G.edges(sort=False)
            [(0, 1, None), (0, 3, None), (0, 4, None), (0, 5, None),
             (0, 6, None), (1, 2, None), (1, 4, None), (2, 3, None),
             (2, 4, 'label'), (2, 5, None), (2, 7, None), (3, 4, None),
             (3, 6, None), (4, 5, None), (4, 6, 'label'), (5, 7, None),
             (6, 7, None)]
            sage: G.add_edges([(4, 7, 'label')])
            sage: G.edges(sort=False)
            [(0, 1, None), (0, 3, None), (0, 4, None), (0, 5, None),
             (0, 6, None), (1, 2, None), (1, 4, None), (2, 3, None),
             (2, 4, 'label'), (2, 5, None), (2, 7, None), (3, 4, None),
             (3, 6, None), (4, 5, None), (4, 6, 'label'), (4, 7, 'label'),
             (5, 7, None), (6, 7, None)]

        Note that the ``weight`` of the edge shall be input as the ``label``::

            sage: G.add_edge((1, 3), label=5)
            sage: G.edges()
            [(0, 1, None), (0, 3, None), (0, 4, None), (0, 5, None),
             (0, 6, None), (1, 2, None), (1, 3, 5), (1, 4, None),
             (2, 3, None), (2, 4, 'label'), (2, 5, None), (2, 7, None),
             (3, 4, None), (3, 6, None), (4, 5, None), (4, 6, 'label'),
             (4, 7, 'label'), (5, 7, None), (6, 7, None)]
            sage: G.add_edge((2, 4, 6), label=6)
            Traceback (most recent call last):
            ...
            ValueError: the graph obtained after the addition of edge
            (((2, 4, 6), None, 6)) is not matching covered

        Vertex name cannot be ``None``, so::

            sage: W = graphs.WheelGraph(6)
            sage: H = MatchingCoveredGraph(W)
            sage: H.add_edge(None, 1)
            Traceback (most recent call last):
            ...
            ValueError: the graph obtained after the addition of edge
            ((None, 1, None)) is not matching covered
            sage: H.edges(sort=False)  # No alteration to the existing graph
            [(0, 1, None), (0, 2, None), (0, 3, None), (0, 4, None),
             (0, 5, None), (1, 2, None), (1, 5, None), (2, 3, None),
             (3, 4, None), (4, 5, None)]
            sage: H.add_edge(None, None)
            Traceback (most recent call last):
            ...
            ValueError: the graph obtained after the addition of edge
            ((None, None, None)) is not matching covered
            sage: H.edges(sort=False)  # No alteration to the existing graph
            [(0, 1, None), (0, 2, None), (0, 3, None), (0, 4, None),
             (0, 5, None), (1, 2, None), (1, 5, None), (2, 3, None),
             (3, 4, None), (4, 5, None)]

        EXAMPLES:

        Adding an already existing edge::

            sage: P = graphs.PetersenGraph()
            sage: G = MatchingCoveredGraph(P)
            sage: G.add_edge(next(G.edge_iterator()))
            sage: P == G
            True
            sage: G.size()
            15
            sage: G.allow_multiple_edges(True)
            sage: G.add_edge(0, 1)
            sage: G.size()
            16

        Adding an edge such that the resulting graph is matching covered::

            sage: P = graphs.PetersenGraph()
            sage: G = MatchingCoveredGraph(P)
            sage: G.add_edge(1, 4)
            sage: G.edges(sort=False)
            [(0, 1, None), (0, 4, None), (0, 5, None), (1, 2, None),
             (1, 4, None), (1, 6, None), (2, 3, None), (2, 7, None),
             (3, 4, None), (3, 8, None), (4, 9, None), (5, 7, None),
             (5, 8, None), (6, 8, None), (6, 9, None), (7, 9, None)]

        Adding an edge with both the incident vertices being existent such
        that the resulting graph is not matching covered::

            sage: C = graphs.CycleGraph(4)
            sage: G = MatchingCoveredGraph(C)
            sage: G.add_edge(0, 2)
            Traceback (most recent call last):
            ...
            ValueError: the graph obtained after the addition of edge
            ((0, 2, None)) is not matching covered
            sage: G.edges(sort=False) # No alteration to the existing graph
            [(0, 1, None), (0, 3, None), (1, 2, None), (2, 3, None)]

        Adding an edge with exactly one incident vertex that is nonexistent
        throws a :exc:`ValueError` exception, as the resulting graph would
        have an odd order::

            sage: C = graphs.CycleGraph(4)
            sage: G = MatchingCoveredGraph(C)
            sage: G.add_edge(0, 4)
            Traceback (most recent call last):
            ...
            ValueError: the graph obtained after the addition of edge
            ((0, 4, None)) is not matching covered
            sage: G.edges(sort=False) # No alteration to the existing graph
            [(0, 1, None), (0, 3, None), (1, 2, None), (2, 3, None)]

        Adding an edge with both the incident vertices that is nonexistent
        throws a :exc:`ValueError` exception, as the resulting graph would
        have been disconnected::

            sage: C = graphs.CycleGraph(4)
            sage: G = MatchingCoveredGraph(C)
            sage: G.add_edge(4, 5)
            Traceback (most recent call last):
            ...
            ValueError: the graph obtained after the addition of edge
            ((4, 5, None)) is not matching covered
            sage: G.edges(sort=False) # No alteration to the existing graph
            [(0, 1, None), (0, 3, None), (1, 2, None), (2, 3, None)]

        Adding a self-loop::

            sage: H = graphs.HeawoodGraph()
            sage: G = MatchingCoveredGraph(H)
            sage: v = next(G.vertex_iterator())
            sage: G.add_edge(v, v)
            Traceback (most recent call last):
            ...
            ValueError: loops are not allowed in matching covered graphs
        """
        if label is None:
            if v is None:
                try:
                    u, v, label = u
                except Exception:
                    try:
                        u, v = u
                    except Exception:
                        pass

        else:
            if v is None:
                try:
                    u, v = u
                except Exception:
                    pass

        if u in self and v in self:
            if u == v:
                raise ValueError('loops are not allowed in '
                                 'matching covered graphs')

            # If (u, v, label) is a multiple edge/ an existing edge
            if self.has_edge(u, v):
                self._backend.add_edge(u, v, label, self._directed)
                return

            # Check if there exists an M-alternating odd uv path starting and
            # ending with edges in self._matching
            from sage.graphs.matching import M_alternating_even_mark
            w = next((b if a == u else a) for a, b, *_ in self.get_matching() if u in (a, b))

            if v in M_alternating_even_mark(self, w, self.get_matching()):
                # There exists a perfect matching containing the edge (u, v, label)
                self._backend.add_edge(u, v, label, self._directed)
                return

        raise ValueError('the graph obtained after the addition of edge '
                         '(%s) is not matching covered' % str((u, v, label)))

    @doc_index('Overwritten methods')
    def add_edges(self, edges, loops=False):
        r"""
        Add edges from an iterable container.

        .. NOTE::

            This method overwrites the
            :meth:`~sage.graphs.generic_graph.GenericGraph.add_edges` method
            to ensure that resultant graph is also matching covered.

        INPUT:

        - ``edges`` -- an iterable of edges, given either as ``(u, v)``
          or ``(u, v, 'label')``. If an edge is provided in the format
          ``(u, v)``, the label is set to ``None``.

        - ``loops`` -- boolean (default: ``False``); note that this shall
          always be set to either ``False`` or ``None`` (since matching covered
          graphs are free of loops), in which case all the loops
          ``(v, v, 'label')`` are removed from the iterator. If ``loops`` is
          set to ``True``, a :exc:`ValueError` is thrown.

        - Please note that all the loops present in the iterator are ignored,
          provided that ``loops`` is set to ``False`` or ``None``.

        OUTPUT:

        - If ``loops`` is set to ``True``, a :exc:`ValueError` is returned.

        - If ``edges`` is provided with a valid format, but addition of the
          edges leave the resulting graph not being matching covered, a
          :exc:`ValueError` is returned without any alteration to the existing
          matching covered graph. If the addition of the edges preserves the
          property of matching covered, then the graph is updated and nothing
          is returned.

        - If ``edges`` is provided in an invalid format, a :exc:`ValueError`
          is returned.

        EXAMPLES:

        Providing with an empty list of edges::

            sage: C = graphs.CycleGraph(6)
            sage: G = MatchingCoveredGraph(C)
            sage: G.add_edges([])
            sage: G == C
            True

        Adding some edges, the incident vertices of each of which are existent,
        such that the resulting graph is matching covered::

            sage: S = graphs.StaircaseGraph(4)
            sage: G = MatchingCoveredGraph(S)
            sage: F = [(0, 4), (2, 4), (4, 6), (4, 7)]
            sage: G.add_edges(F)
            sage: G.edges(sort=True, sort_vertices=True)
            [(0, 1, None), (0, 3, None), (0, 4, None), (0, 6, None),
             (1, 2, None), (1, 4, None), (2, 4, None), (2, 5, None),
             (2, 7, None), (3, 4, None), (3, 6, None), (4, 5, None),
             (4, 6, None), (4, 7, None), (5, 7, None), (6, 7, None)]

        Adding some edges, at least one of the incident vertices of some of
        which are nonexistent such that the resulting graph is matching
        covered::

            sage: C = graphs.CycleGraph(8)
            sage: G = MatchingCoveredGraph(C)
            sage: F = [(0, 9), (1, 8), (2, 9), (3, 8),
            ....:      (4, 9), (5, 8), (6, 9), (7, 8)]
            sage: G.add_edges(F)
            sage: G.edges(sort=True, sort_vertices=True)
            [(0, 1, None), (0, 7, None), (0, 9, None), (1, 2, None),
             (1, 8, None), (2, 3, None), (2, 9, None), (3, 4, None),
             (3, 8, None), (4, 5, None), (4, 9, None), (5, 6, None),
             (5, 8, None), (6, 7, None), (6, 9, None), (7, 8, None)]
            sage: G.is_isomorphic(graphs.BiwheelGraph(5))
            True

        Adding a removable double ear to a matching covered graph::

            sage: H = graphs.HexahedralGraph()
            sage: G = MatchingCoveredGraph(H)
            sage: F = {(0, 8, None), (1, 10), (4, 11, 'label'),
            ....:      (5, 9), (8, 9), (10, 11)}
            sage: G.add_edges(F)
            sage: G.edges(sort=True, sort_vertices=True)
            [(0, 1, None), (0, 3, None), (0, 4, None), (0, 8, None),
             (1, 2, None), (1, 5, None), (1, 10, None), (2, 3, None),
             (2, 6, None), (3, 7, None), (4, 5, None), (4, 7, None),
             (4, 11, 'label'), (5, 6, None), (5, 9, None), (6, 7, None),
             (8, 9, None), (10, 11, None)]

        Adding some edges, the incident vertices of each of which are existent,
        such that the resulting graph is NOT matching covered::

            sage: C = graphs.CycleGraph(6)
            sage: G = MatchingCoveredGraph(C)
            sage: F = [(0, 2), (3, 5)]
            sage: G.add_edges(F)
            Traceback (most recent call last):
            ...
            ValueError: the resulting graph after the addition ofthe edges is not matching covered

        Adding some edges, at least one of the incident vertices of some of
        which are nonexistent such that the resulting graph is NOT matching
        covered::

            sage: H = graphs.HexahedralGraph()
            sage: G = MatchingCoveredGraph(H)
            sage: F = [(3, 8), (6, 9), (8, 9)]
            sage: G.add_edges(F)
            Traceback (most recent call last):
            ...
            ValueError: the resulting graph after the addition ofthe edges is not matching covered
            sage: I = [(0, 8), (1, 9)]
            sage: G.add_edges(I)
            Traceback (most recent call last):
            ...
            ValueError: the resulting graph after the addition ofthe edges is not matching covered
            sage: J = [(u, 8) for u in range(8)]
            sage: G.add_edges(J)
            Traceback (most recent call last):
            ...
            ValueError: odd order is not allowed for matching covered graphs

        Setting the parameter ``loops`` to either ``False`` or ``None``::

            sage: W = graphs.WheelGraph(6)
            sage: G = MatchingCoveredGraph(W)
            sage: F = [(0, 0), (1, 3), (2, 4)]
            sage: G.add_edges(edges=F, loops=False)
            sage: G.edges(sort=True, sort_vertices=True)
            [(0, 1, None), (0, 2, None), (0, 3, None), (0, 4, None),
             (0, 5, None), (1, 2, None), (1, 3, None), (1, 5, None),
             (2, 3, None), (2, 4, None), (3, 4, None), (4, 5, None)]
            sage: J = [(1, 1), (3, 5)]
            sage: G.add_edges(edges=J, loops=True)
            Traceback (most recent call last):
            ...
            ValueError: loops are not allowed in matching covered graphs
            sage: G.edges(sort=True, sort_vertices=True)
            [(0, 1, None), (0, 2, None), (0, 3, None), (0, 4, None),
             (0, 5, None), (1, 2, None), (1, 3, None), (1, 5, None),
             (2, 3, None), (2, 4, None), (3, 4, None), (4, 5, None)]

        Setting the parameter ``loops`` to ``True``::

            sage: P = graphs.PetersenGraph()
            sage: G = MatchingCoveredGraph(P)
            sage: F = [(0, 0), (0, 2), (0, 3)]
            sage: G.add_edges(edges=F, loops=True)
            Traceback (most recent call last):
            ...
            ValueError: loops are not allowed in matching covered graphs

        Adding a multiple edge::

            sage: S = graphs.StaircaseGraph(4)
            sage: G = MatchingCoveredGraph(S)
            sage: G.allow_multiple_edges(True)
            sage: F = [(0, 1, 'label'), (0, 4), (1, 2)]
            sage: G.add_edges(F)
            sage: G.edges(sort=False)
            [(0, 1, None), (0, 1, 'label'), (0, 3, None), (0, 4, None),
             (0, 6, None), (1, 2, None), (1, 2, None), (1, 4, None),
             (2, 5, None), (2, 7, None), (3, 4, None), (3, 6, None),
             (4, 5, None), (5, 7, None), (6, 7, None)]
            sage: H = [(0, 1)] * 4
            sage: G.add_edges(H)
            sage: G.edge_label(0, 1)
            [None, None, None, None, None, 'label']

        TESTS:

        Providing with a non-iterable of edges::

            sage: K2 = graphs.CompleteGraph(2)
            sage: G = MatchingCoveredGraph(K2)
            sage: G.add_edges(1234)
            Traceback (most recent call last):
            ...
            ValueError: expected an iterable of edges,
            but got a non-iterable object

        Providing with an edge in ``edges`` that has 0 values to unpack::

            sage: W = graphs.WagnerGraph()
            sage: G = MatchingCoveredGraph(W)
            sage: G.add_edges([()])
            Traceback (most recent call last):
            ...
            ValueError: need more than 1 value to unpack for edge: ()

        Providing with an edge in ``edges`` that has precisely one value to unpack::

            sage: T = graphs.TruncatedBiwheelGraph(10)
            sage: G = MatchingCoveredGraph(T)
            sage: G.add_edges([(0, )])
            Traceback (most recent call last):
            ...
            ValueError: need more than 1 value to unpack for edge: (0,)

        Providing with an edge in ``edges`` that has more than 3 values to unpack::

            sage: B = graphs.BiwheelGraph(5)
            sage: G = MatchingCoveredGraph(B)
            sage: G.add_edges([(0, 1, 2, 3, 4)])
            Traceback (most recent call last):
            ...
            ValueError: too many values to unpack (expected 2) for edge: (0, 1, 2, 3, 4)

        Providing with an edge of unknown data type::

            sage: M = graphs.MurtyGraph()
            sage: G = MatchingCoveredGraph(M)
            sage: F = [None, 'edge', None]
            sage: G.add_edges(F)
            Traceback (most recent call last):
            ...
            TypeError: input edge None is of unknown type
        """
        if loops:
            raise ValueError('loops are not allowed in '
                             'matching covered graphs')

        if not edges:  # do nothing
            return

        from collections.abc import Iterable
        if not isinstance(edges, Iterable):
            raise ValueError('expected an iterable of edges, '
                             'but got a non-iterable object')

        links = []  # to extract the nonloop input edges
        for edge in edges:
            if hasattr(edge, '__len__'):
                if len(edge) <= 1:
                    raise ValueError('need more than 1 value to unpack '
                                     f'for edge: {edge}')

                elif len(edge) > 3:
                    raise ValueError('too many values to unpack (expected 2) '
                                     f'for edge: {edge}')

            else:
                raise TypeError(f'input edge {edge} is of unknown type')

            u, v, l = None, None, None

            if len(edge) == 2:
                u, v = edge
            else:
                u, v, l = edge

            if u != v:
                links.append((u, v, l))

        # If each of the input edges is existent
        if (self.allows_multiple_edges()
            and all(self.has_edge(*edge) for edge in links)):
            self._backend.add_edges(links, self._directed)
            return

        # Check if all the incident vertices of the input edges are existent
        new_vertices = {x for u, v, _ in links for x in (u, v)
                        if x not in self}

        # Throw error if the no. of new vertices is odd
        if len(new_vertices) % 2:
            raise ValueError('odd order is not allowed for '
                             'matching covered graphs')

        try:
            G = Graph(self, multiedges=self.allows_multiple_edges())
            G.add_edges(edges=links, loops=loops)

            # Check if G has a vertex with at most 1 neighbor
            if any(len(G.neighbors(v)) <= 1 for v in G):
                raise ValueError('the resulting graph after the addition of'
                                 'the edges is not matching covered')

            # If all the vertices are existent, the existing perfect matching
            # can be used.
            if not new_vertices:
                self.__init__(data=G, matching=self.get_matching())

            else:
                # Check if the existing perfect matching may be extended to a
                # perfect matching of the new graph
                links_with_two_new_vertices = []

                for edge in links:
                    if edge[0] in new_vertices and edge[1] in new_vertices:
                        links_with_two_new_vertices.append(edge)

                M = Graph(data=links_with_two_new_vertices, format='list_of_edges')
                M.add_edges(self.get_matching())

                # Check if M is a perfect matching of the resulting graph
                if (G.order() != 2*M.size()):
                    M = None

                self.__init__(data=G, matching=M)

        except Exception:
            raise ValueError('the resulting graph after the addition of'
                             'the edges is not matching covered')

    @doc_index('Overwritten methods')
    def add_vertex(self, name=None):
        r"""
        Add a vertex to the (matching covered) graph.

        .. NOTE::

            This method overwrites the
            :meth:`~sage.graphs.generic_graph.GenericGraph.add_vertex` method
            to ensure that isolated vertices are forbidden in
            :class:`~MatchingCoveredGraph`.

        INPUT:

        - ``name`` -- an immutable object (default: ``None``); when no name is
          specified (default), then the new vertex will be represented by the
          least integer not already representing a vertex. ``name`` must be an
          immutable object (e.g., an integer, a tuple, etc.).

        OUTPUT:

        - If ``name`` specifies an existing vertex, then nothing is done.
          Otherwise a :exc:`ValueError` is returned with no change to the
          existing (matching covered) graph is returned since matching covered
          graphs are free of isolated vertices.

        EXAMPLES:

        Adding an existing vertex::

            sage: P = graphs.PetersenGraph()
            sage: P
            Petersen graph: Graph on 10 vertices
            sage: G = MatchingCoveredGraph(P)
            sage: G
            Matching covered petersen graph: graph on 10 vertices
            sage: u = next(G.vertex_iterator())
            sage: G.add_vertex(u)
            sage: G
            Matching covered petersen graph: graph on 10 vertices

        Adding a new/ non-existing vertex::

            sage: G.add_vertex()
            Traceback (most recent call last):
            ...
            ValueError: isolated vertices are not allowed in matching covered graphs
            sage: u = 100
            sage: G.add_vertex(u)
            Traceback (most recent call last):
            ...
            ValueError: isolated vertices are not allowed in matching covered graphs
        """
        if name not in self:
            raise ValueError('isolated vertices are not allowed in '
                             'matching covered graphs')

    @doc_index('Overwritten methods')
    def add_vertices(self, vertices):
        r"""
        Add vertices to the (matching covered) graph from an iterable container
        of vertices.

        .. NOTE::

            This method overwrites the
            :meth:`~sage.graphs.generic_graph.GenericGraph.add_vertices` method
            to ensure that isolated vertices are forbidden in
            :class:`~MatchingCoveredGraph`.

        INPUT:

        - ``vertices`` -- iterator container of vertex labels. A new label is
          created, used and returned in the output list for all ``None`` values
          in ``vertices``.

        OUTPUT:

        - If all of the vertices are existing vertices of the (matching
          covered) graph, then nothing is done; otherwise a :exc:`ValueError`
          is returned with no change to the existing (matching covered) graph
          since matching covered graphs are free of isolated vertices.

        EXAMPLES:

        Adding a list of already existing vertices::

            sage: T = graphs.TruncatedBiwheelGraph(15)
            sage: T
            Truncated biwheel graph: Graph on 30 vertices
            sage: G = MatchingCoveredGraph(T)
            sage: G
            Matching covered truncated biwheel graph: graph on 30 vertices
            sage: S = [0, 1, 2, 3]  # We choose 4 existing vertices
            sage: G.add_vertices(S)
            sage: G
            Matching covered truncated biwheel graph: graph on 30 vertices

        Adding a list of vertices in which at least one is non-existent or
        ``None`` or possibly both::

            sage: T = graphs.CompleteGraph(2)
            sage: T
            Complete graph: Graph on 2 vertices
            sage: G = MatchingCoveredGraph(T)
            sage: G
            Matching covered complete graph: graph on 2 vertices
            sage: S1 = [2, 3, 4]
            sage: G.add_vertices(S1)
            Traceback (most recent call last):
            ...
            ValueError: isolated vertices are not allowed in matching covered graphs
            sage: S2 = [None, None]
            sage: G.add_vertices(S2)
            Traceback (most recent call last):
            ...
            ValueError: isolated vertices are not allowed in matching covered graphs
            sage: S3 = [2, None, None, 5]
            sage: G.add_vertices(S3)
            Traceback (most recent call last):
            ...
            ValueError: isolated vertices are not allowed in matching covered graphs
        """
        if any(vertex not in self for vertex in vertices):
            raise ValueError('isolated vertices are not allowed in '
                             'matching covered graphs')

    @doc_index('Overwritten methods')
    def allow_loops(self, new, check=True):
        r"""
        Change whether loops are allowed in (matching covered) graphs.

        .. NOTE::

            This method overwrites the
            :meth:`~sage.graphs.generic_graph.GenericGraph.allow_loops` method
            to ensure that loops are forbidden in :class:`~MatchingCoveredGraph`.

        INPUT:

        - ``new`` -- boolean

        - ``check`` -- boolean (default: ``True``); whether to remove existing
          loops from the graph when the new status is ``False``. It is an
          argument in
          :meth:`~sage.graphs.generic_graph.GenericGraph.allow_loops` method
          and is not used in this overwritten one.

        OUTPUT:

        - A :exc:`ValueError` is returned with no change to the existing
          (matching covered) graph if ``new`` is ``True`` since a matching
          covered graph, by definition, is free of self-loops. If ``new`` is
          set to ``False``, there is no output.

        EXAMPLES:

        Petersen graph is matching covered::

            sage: P = graphs.PetersenGraph()
            sage: P.is_matching_covered()
            True
            sage: G = MatchingCoveredGraph(P)
            sage: G.allow_loops(True)
            Traceback (most recent call last):
            ...
            ValueError: loops are not allowed in matching covered graphs

        .. SEEALSO::

            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.allows_loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.has_loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.loop_edges`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.loop_vertices`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.number_of_loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.remove_loops`
        """
        if new:
            raise ValueError('loops are not allowed in '
                             'matching covered graphs')

    @doc_index('Overwritten methods')
    def allows_loops(self):
        r"""
        Return whether loops are permitted in (matching covered) graphs.

        .. NOTE::

            This method overwrites the
            :meth:`~sage.graphs.generic_graph.GenericGraph.allows_loops` method
            to show that loops are forbidden in :class:`~MatchingCoveredGraph`.

        OUTPUT:

        - A boolean value ``False`` is returned, since matching covered graphs,
          by definition, are free of loops.

        EXAMPLES:

        Petersen graph is matching covered::

            sage: P = graphs.PetersenGraph()
            sage: P.is_matching_covered()
            True
            sage: G = MatchingCoveredGraph(P)
            sage: G.allows_loops()
            False

        .. SEEALSO::

            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.allow_loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.has_loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.loop_edges`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.loop_vertices`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.number_of_loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.remove_loops`
        """
        return False

    @doc_index('Barriers and canonical partition')
    def canonical_partition(self):
        r"""
        Return the canonical partition of the (matching covered) graph.

        For a matching covered graph `G`, a subset `B` of the vertex set `V` is
        a barrier if `|B| = o(G - B)`, where `|B|` denotes the cardinality of
        the set `B` and `o(G - B)` denotes the number of odd components in the
        graph `G - B`. And a barrier `B` is a maximal barrier if `C` is not a
        barrier for each `C` such that `B \subset C \subseteq V`.

        Note that in a matching covered graph, each vertex belongs to a unique
        maximal barrier. The maximal barriers of a matching covered graph
        partitions its vertex set and the partition of the vertex set of a
        matching covered graph into its maximal barriers is called as its
        *canonical* *partition*.

        OUTPUT:

        - A list of sets that constitute a (canonical) partition of the vertex
          set, wherein each set is a (unique) maximal barrier of the (matching
          covered) graph.

        EXAMPLES:

        Show the maximal barrier of the graph `K_4 \odot K_{3, 3}`::

            sage: G = Graph([
            ....:    (0, 2), (0, 3), (0, 4), (1, 2),
            ....:    (1, 3), (1, 4), (2, 5), (3, 6),
            ....:    (4, 7), (5, 6), (5, 7), (6, 7)
            ....: ])
            sage: H = MatchingCoveredGraph(G)
            sage: H.canonical_partition()
            [{0}, {1}, {2, 3, 4}, {5}, {6}, {7}]

        For a bicritical graph (for instance, the Petersen graph), the
        canonical partition constitutes of only singleton sets each containing
        an individual vertex::

            sage: P = graphs.PetersenGraph()
            sage: G = MatchingCoveredGraph(P)
            sage: G.canonical_partition()
            [{0}, {1}, {2}, {3}, {4}, {5}, {6}, {7}, {8}, {9}]

        For a bipartite matching covered graph (for instance, the Hexahedral
        graph), the canonical partition consists of two sets each of which
        corresponds to the individual color class::

            sage: H = graphs.HexahedralGraph()
            sage: G = MatchingCoveredGraph(H)
            sage: G.canonical_partition()
            [{0, 2, 5, 7}, {1, 3, 4, 6}]
            sage: B = BipartiteGraph(H)
            sage: list(B.bipartition()) == G.canonical_partition()
            True

        REFERENCES:

            - [LM2024]_

        .. SEEALSO::

            - :meth:`~sage.graphs.graph.Graph.is_bicritical`
            - :meth:`~sage.graphs.graph.Graph.is_matching_covered`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.maximal_barrier`
        """
        visited = set()

        maximal_barriers = []
        for v in self:
            if v not in visited:
                B = self.maximal_barrier(v)
                visited.update(B)
                maximal_barriers.append(B)

        return maximal_barriers

    @doc_index('Overwritten methods')
    def delete_vertex(self, vertex, in_order=False):
        r"""
        Delete a vertex, removing all incident edges.

        .. NOTE::

            This method overwrites the
            :meth:`~sage.graphs.generic_graph.GenericGraph.delete_vertex`
            method to ensure that an odd order is forbidden in
            :class:`~MatchingCoveredGraph`.

        INPUT:

        - ``vertex`` -- a vertex that is to be deleted.

        - ``in_order`` -- boolean (default: ``False``); if ``True``, this
          deletes the `i`-th vertex in the sorted list of vertices, i.e.
          ``G.vertices(sort=True)[i]``

        OUTPUT:

        - Deleting a non-existent vertex raises a :exc:`ValueError` exception;
          also a (different) :exc:`ValueError` is returned on deleting an
          existing vertex since matching covered graphs are of even order. In
          both cases no modifications are made to the existing (matching
          covered) graph.

        EXAMPLES:

        Deleting a non-existent vertex::

            sage: W = graphs.WheelGraph(12)
            sage: G = MatchingCoveredGraph(W)
            sage: u = 100
            sage: G.delete_vertex(u)
            Traceback (most recent call last):
            ...
            ValueError: vertex (100) not in the graph
            sage: G.delete_vertex(vertex=u, in_order=True)
            Traceback (most recent call last):
            ...
            ValueError: vertex (100) not in the graph

        Deleting an existing vertex::

            sage: W = graphs.WheelGraph(12)
            sage: G = MatchingCoveredGraph(W)
            sage: u = next(G.vertex_iterator())
            sage: G.delete_vertex(u)
            Traceback (most recent call last):
            ...
            ValueError: odd order is not allowed for matching covered graphs
            sage: G.delete_vertex(vertex=u, in_order=True)
            Traceback (most recent call last):
            ...
            ValueError: odd order is not allowed for matching covered graphs
        """
        if vertex not in self:
            raise ValueError('vertex (%s) not in the graph' % str(vertex))

        if in_order:
            vertex = self.vertices(sort=True)[vertex]

        raise ValueError('odd order is not allowed for '
                         'matching covered graphs')

    @doc_index('Overwritten methods')
    def delete_vertices(self, vertices):
        r"""
        Delete specified vertices form ``self``.

        This method deletes the vertices from the iterable container
        ``vertices`` from ``self`` along with incident edges. An error is
        raised if the resulting graph is not matching covered.

        .. NOTE::

            This method overwrites the
            :meth:`~sage.graphs.generic_graph.GenericGraph.delete_vertices`
            method to ensure that an odd order is forbidden in
            :class:`~MatchingCoveredGraph`.

        INPUT:

        - ``vertices`` -- a list/ set of vertices that are to be deleted.

        OUTPUT:

        - Deleting a non-existent vertex will raise a :exc:`ValueError`
          exception, in which case none of the vertices in ``vertices``
          is deleted.

        - If all of the vertices in the list/ set provided exist in the graph,
          but the resulting graph after deletion of all of those is not
          matching covered, then a :exc:`ValueError` exception is raised
          without any alterations to the existing (matching covered) graph,
          otherwise the vertices are deleted and nothing is returned.

        EXAMPLES:

        Providing with an empty list of vertices::

            sage: C = graphs.CycleGraph(6)
            sage: G = MatchingCoveredGraph(C)
            sage: G.delete_vertices([])
            sage: G == C
            True

        Removing all the existent vertices::

            sage: M = graphs.MoebiusLadderGraph(10)
            sage: G = MatchingCoveredGraph(M)
            sage: S = list(G.vertices())
            sage: G.delete_vertices(S)
            Traceback (most recent call last):
            ...
            ValueError: the resulting graph after the removal of the vertices
            is trivial, therefore is not matching covered

        Providing with a list of vertices with at least one non-existent
        vertex::

            sage: S = graphs.StaircaseGraph(4)
            sage: S
            Staircase graph: Graph on 8 vertices
            sage: G = MatchingCoveredGraph(S)
            sage: G
            Matching covered staircase graph: graph on 8 vertices
            sage: T = list(range(5, 20, 2))
            sage: G.delete_vertices(T)
            Traceback (most recent call last):
            ...
            ValueError: vertex (9) not in the graph

        Removing an odd no. of distinct vertices from
        a matching covered graph::

            sage: P = graphs.PetersenGraph()
            sage: G = MatchingCoveredGraph(P)
            sage: S = [0, 1, 2, 10, 10, 100]
            sage: G.delete_vertices(S)
            Traceback (most recent call last):
            ...
            ValueError: an odd no. of distinct vertices can not be
            removed from a matching covered graph

        Providing with a list of existent vertices whose deletion results in a
        graph which is not matching covered::

            sage: S = graphs.StaircaseGraph(4)
            sage: S
            Staircase graph: Graph on 8 vertices
            sage: G = MatchingCoveredGraph(S)
            sage: G
            Matching covered staircase graph: graph on 8 vertices
            sage: T = [1, 4]
            sage: G.delete_vertices(T)
            Traceback (most recent call last):
            ...
            ValueError: the resulting graph after the removal of
            the vertices is not matching covered

        Providing with a list of existent vertices after the deletion of which
        the resulting graph is still matching covered; note that in the
        following example, after the deletion of two vertices from a staircase
        graph, the resulting graph is NOT a staircase graph
        (see :issue:`38768`)::

            sage: S = graphs.StaircaseGraph(4)
            sage: S
            Staircase graph: Graph on 8 vertices
            sage: G = MatchingCoveredGraph(S)
            sage: G
            Matching covered staircase graph: graph on 8 vertices
            sage: T = [6, 7]
            sage: G.delete_vertices(T)
            sage: G  # Matching covered graph on 6 vertices
            Matching covered staircase graph: graph on 6 vertices
        """
        if not vertices:   # do nothing
            return

        # Remove potentially duplicated vertices
        vertices = set(vertices)

        if len(vertices) % 2:  # try to remove an odd number of vertices
            raise ValueError('an odd no. of distinct vertices can not be '
                             'removed from a matching covered graph')

        for vertex in vertices:
            if vertex not in self:
                raise ValueError('vertex (%s) not in the graph' % str(vertex))

        if self.order() == len(vertices):
            raise ValueError('the resulting graph after the removal of the '
                             'vertices is trivial, therefore is not '
                             'matching covered')

        try:
            G = Graph(self, multiedges=self.allows_multiple_edges())
            G.delete_vertices(vertices)

            M = Graph(self.get_matching())

            M.delete_vertices(vertices)
            # The resulting matching after the removal of the input vertices
            # must be a valid perfect matching of the resulting graph obtained
            # after the removal of the vertices

            if (G.order() != 2*M.size()):
                M = None

            self.__init__(data=G, matching=M)

        except Exception:
            raise ValueError('the resulting graph after the removal of '
                             'the vertices is not matching covered')

    @doc_index('Miscellaneous methods')
    def get_matching(self):
        r"""
        Return an :class:`~EdgesView` of ``self._matching``.

        OUTPUT:

        - This method returns :class:`EdgesView` of the edges of a
          perfect matching of the (matching covered) graph.

        EXAMPLES:

        If one specifies a perfect matching while initializing the object, the
        value of ``self._matching`` is the same matching::

            sage: P = graphs.PetersenGraph()
            sage: M = [(0, 1), (2, 3), (4, 9), (5, 7), (6, 8)]
            sage: G = MatchingCoveredGraph(P, M)
            sage: sorted(G.get_matching())
            [(0, 1), (2, 3), (4, 9), (5, 7), (6, 8)]
            sage: M == sorted(G.get_matching())
            True

        If no matching is specified while initializing a matching
        covered graph, a perfect matching is computed
        :meth:`~sage.graphs.graph.Graph.matching` and that is captured as
        ``self._matching``::

            sage: P = graphs.PetersenGraph()
            sage: M = P.matching()
            sage: G = MatchingCoveredGraph(P)
            sage: sorted(G.get_matching())
            [(0, 5, None), (1, 6, None), (2, 7, None), (3, 8, None), (4, 9, None)]
            sage: sorted(M) == sorted(G.get_matching())
            True
        """
        return self._matching

    @doc_index('Barriers and canonical partition')
    def maximal_barrier(self, vertex):
        r"""
        Return the (unique) maximal barrier containing the vertex.

        For a matching covered graph `G`, a subset `B` of the vertex set `V` is
        a barrier if `|B| = o(G - B)`, where `|B|` denotes the cardinality of
        the set `B` and `o(G - B)` denotes the number of odd components in the
        graph `G - B`. And a barrier `B` is a maximal barrier if `C` is not a
        barrier for each `C` such that `B \subset C \subseteq V`.

        In a matching covered graph, each vertex belongs to a unique maximal
        barrier, which is a consequence of the following theorem.

        .. RUBRIC:: Theorem [LM2024]_:

        Let `u` and `v` be any two vertices in a matchable graph `G`. Then the
        graph `G - u - v` is matchable if and only if there is no barrier of
        `G` which contains both `u` and `v`.

        And in order to find the vertices that do not lie in the maximal
        barrier containing the provided vertex in linear time we take
        inspiration of the `M` alternating tree seach method [LR2004]_.

        INPUT:

        - ``vertex`` -- a vertex of the graph

        OUTPUT:

        - A :exc:`~ValueError` is returned if ``vertex`` is not a vertex of the
          graph, otherwise a set of vertices that constitute the (unique)
          maximal barrier containing the vertex is returned.

        EXAMPLES:

        The graph `K_4 \odot K_{3, 3}` is matching covered. Show the set of
        vertices in the (unique) maximal barrier containing the vertex `2`::

            sage: G = Graph([
            ....:    (0, 2), (0, 3), (0, 4), (1, 2),
            ....:    (1, 3), (1, 4), (2, 5), (3, 6),
            ....:    (4, 7), (5, 6), (5, 7), (6, 7)
            ....: ])
            sage: H = MatchingCoveredGraph(G)
            sage: B = H.maximal_barrier(2)
            sage: B
            {2, 3, 4}

        Let `B` be a maximal barrier of a matching covered graph `G` (which is,
        of course, a matchable graph). The graph, `J := G - B` has no even
        component::

            sage: J = G.copy()
            sage: J.delete_vertices(B)
            sage: all(len(K)%2 != 0 for K in J.connected_components(sort=True))
            True

        Let `B` be a maximal barrier in a matching covered graph `G` and let
        `M` be a perfect matching of `G`. If `K` is an odd component of
        `J := G - B`, then `M \cap \partial_G(K)` has precisely one edge and if
        `v` is the end of that edge in `V(K)`, then `M \cap E(K)` is a perfect
        matching of `K - v`::

            sage: K = J.subgraph(vertices=(J.connected_components(sort=True))[0])
            sage: # Let F := \partial_G(K) and T := M \cap F
            sage: F = [edge for edge in G.edge_iterator()
            ....:      if (edge[0] in K and edge[1] not in K)
            ....:      or (edge[0] not in K and edge[1] in K)
            ....: ]
            sage: M = H.get_matching()
            sage: T = [edge for edge in F if edge in M]
            sage: len(T) == 1
            True
            sage: v = T[0][0] if T[0][0] in K else T[0][1]
            sage: # Let N := M \cap E(K) and L := K - v
            sage: N = Graph([edge for edge in K.edge_iterator() if edge in M])
            sage: L = K.copy()
            sage: L.delete_vertex(v)
            sage: # Check if N is a perfect matching of L
            sage: L.order() == 2*N.size()
            True

        Let `B` be a maximal barrier of a matching covered graph `G` (which is,
        of course, a matchable graph). The graph induced by each component of
        `G - B` is factor critical::

            sage: all((K.subgraph(vertices=connected_component)).is_factor_critical()
            ....:     for connected_component in K.connected_components(sort=True)
            ....: )
            True

        For a bicritical graph (for instance, the Petersen graph), for each
        vertex the maximal barrier is a singleton set containing only that
        vertex::

            sage: P = graphs.PetersenGraph()
            sage: G = MatchingCoveredGraph(P)
            sage: u = 0
            sage: set([u]) == G.maximal_barrier(u)
            True

        In a bipartite matching covered graph (for instance, the Hexahedral
        graph), for a vertex, the maximal barrier is the set of vertices of
        the color class that the particular vertex belongs to. In other words,
        there are precisely two maximal barriers in a bipartite matching
        covered graph, that is, the vertex sets of the individual color class::

            sage: G = graphs.HexahedralGraph()
            sage: H = MatchingCoveredGraph(G)
            sage: A, _ = H.bipartite_sets()
            sage: # needs random
            sage: import random
            sage: a = random.choice(list(A))
            sage: A == H.maximal_barrier(a)
            True

        Maximal barriers of matching covered graph constitute a partition of
        its vertex set::

            sage: S = set()
            sage: for v in H:
            ....:     B = tuple(sorted(list(H.maximal_barrier(v))))
            ....:     S.add(B)
            sage: S = list(S)
            sage: # Check that S is a partition of the vertex set of H
            sage: # Part 1: Check if S spans the vertex set of H
            sage: sorted([u for B in S for u in B]) == sorted(list(H))
            True
            sage: # Part 2: Check if each maximal barrier in S is disjoint
            sage: is_disjoint = True
            sage: for i in range(len(S)):
            ....:     for j in range(i+1, len(S)):
            ....:         c = [v for v in S[i] if v in S[j]]
            ....:         is_disjoint = (len(c) == 0)
            sage: is_disjoint
            True

        TESTS:

        Providing with a nonexistent vertex::

            sage: P = graphs.PetersenGraph()
            sage: G = MatchingCoveredGraph(P)
            sage: G.maximal_barrier('')
            Traceback (most recent call last):
            ...
            ValueError: vertex  not in the graph
            sage: G.maximal_barrier(100)
            Traceback (most recent call last):
            ...
            ValueError: vertex 100 not in the graph

        REFERENCES:

        - [LZ2004]_
        - [LM2024]_

        .. SEEALSO::

            - :meth:`~sage.graphs.graph.Graph.is_bicritical`
            - :meth:`~sage.graphs.graph.Graph.is_matching_covered`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.canonical_partition`
        """
        if vertex not in self:
            raise ValueError('vertex {} not in the graph'.format(vertex))

        # u: The M neighbor of vertex
        matching = self.get_matching()
        u = next((a if b == vertex else b) for a, b, *_ in matching if vertex in [a, b])

        # Goal: Find the vertices w such that G - w - vertex is matchable.
        # In other words, there exists an odd length M-alternating vertex-w
        # path in G, starting and ending with edges in M. Alternatively, there
        # exists an even length M-alternating u-w path in the graph G - vertex
        # starting with an edge not in M and ending with and edge in M.

        # even: The set of all such vertex w
        from sage.graphs.matching import M_alternating_even_mark
        even = M_alternating_even_mark(G=self, matching=matching,
                                       vertex=u)

        B = set([vertex])
        B.update(v for v in self if v not in even)

        return B

    @doc_index('Overwritten methods')
    def has_loops(self) -> bool:
        r"""
        Check whether there are loops in the (matching covered) graph.

        .. NOTE::

            This method overwrites the
            :meth:`~sage.graphs.generic_graph.GenericGraph.has_loops` method in
            order to return ``False`` as matching covered graphs are always
            free of looped edges.

        OUTPUT:

        - A boolean ``False`` is returned since matching covered graphs, by
          definition, are free of self-loops.

        EXAMPLES:

        A matching covered graph, for instance the Petersen graph, is always free
        of loops::

            sage: P = graphs.PetersenGraph()
            sage: G = MatchingCoveredGraph(P)
            sage: G
            Matching covered petersen graph: graph on 10 vertices
            sage: G.has_loops()
            False
            sage: G.allows_loops()
            False
            sage: G.add_edge(0, 0)
            Traceback (most recent call last):
            ...
            ValueError: loops are not allowed in matching covered graphs

        A matching covered graph may support multiple edges, still no
        loops are allowed::

            sage: K = graphs.CompleteGraph(2)
            sage: G = MatchingCoveredGraph(K)
            sage: G.allow_multiple_edges(True)
            sage: G
            Matching covered complete graph: multi-graph on 2 vertices
            sage: G.add_edge(0, 1, 'label')
            sage: G.add_edge(0, 0)
            Traceback (most recent call last):
            ...
            ValueError: loops are not allowed in matching covered graphs
            sage: G.edges(sort=False)
            [(0, 1, None), (0, 1, 'label')]
            sage: G.allows_loops()
            False
            sage: G.has_loops()
            False
            sage: G.allow_loops(True)
            Traceback (most recent call last):
            ...
            ValueError: loops are not allowed in matching covered graphs

        .. SEEALSO::

            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.allow_loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.allows_loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.loop_edges`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.loop_vertices`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.number_of_loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.remove_loops`
        """
        return False

    @doc_index('Overwritten methods')
    def has_perfect_matching(G, algorithm='Edmonds', solver=None, verbose=0,
                             *, integrality_tolerance=1e-3):
        r"""
        Check whether the graph has a perfect matching.

        .. NOTE::

            This method overwrites the
            :meth:`~sage.graphs.graph.Graph.has_perfect_matching` method in
            order to return ``True`` (provided the input arguments are valid)
            as matching covered graphs always admit a perfect matching.

        INPUT:

        - ``algorithm`` -- string (default: ``'Edmonds'``)

          - ``'Edmonds'`` uses Edmonds' algorithm as implemented in NetworkX to
            find a matching of maximal cardinality, then check whether this
            cardinality is half the number of vertices of the graph.

          - ``'LP_matching'`` uses a Linear Program to find a matching of
            maximal cardinality, then check whether this cardinality is half the
            number of vertices of the graph.

          - ``'LP'`` uses a Linear Program formulation of the perfect matching
            problem: put a binary variable ``b[e]`` on each edge `e`, and for
            each vertex `v`, require that the sum of the values of the edges
            incident to `v` is 1.

        - ``solver`` -- string (default: ``None``); specifies a Mixed Integer
          Linear Programming (MILP) solver to be used. If set to ``None``, the
          default one is used. For more information on MILP solvers and which
          default solver is used, see the method :meth:`solve
          <sage.numerical.mip.MixedIntegerLinearProgram.solve>` of the class
          :class:`MixedIntegerLinearProgram
          <sage.numerical.mip.MixedIntegerLinearProgram>`.

        - ``verbose`` -- integer (default: 0); sets the level of verbosity:
          set to 0 by default, which means quiet (only useful when
          ``algorithm == "LP_matching"`` or ``algorithm == "LP"``)

        - ``integrality_tolerance`` -- float; parameter for use with MILP
          solvers over an inexact base ring; see
          :meth:`MixedIntegerLinearProgram.get_values`.

        OUTPUT:

        - If the input arguments are valid, a boolean (``True``) is returned as
          a maximum matching of a matching covered graph is always a perfect
          matching, otherwise a :exc:`~ValueError` is raised.

        EXAMPLES:

        Note that regardless of the algorithm (as long as the input arguments
        are in valid format), the method always returns the boolean ``True``::

            sage: P = graphs.PetersenGraph()
            sage: P.has_perfect_matching()  # Calls Graph.has_perfect_matching()
            True
            sage: G = MatchingCoveredGraph(P)
            sage: G.has_perfect_matching()  # Calls MatchingCoveredGraph.has_perfect_matching()
            True
            sage: W = graphs.WheelGraph(6)
            sage: H = MatchingCoveredGraph(W)
            sage: H.has_perfect_matching(algorithm='LP_matching')
            True

        Providing with an algorithm, that is not one of ``'Edmonds'``,
        ``'LP_matching'`` or ``'LP'``::

            sage: S = graphs.StaircaseGraph(4)
            sage: J = MatchingCoveredGraph(S)
            sage: J.has_perfect_matching(algorithm='algorithm')
            Traceback (most recent call last):
            ...
            ValueError: algorithm must be set to 'Edmonds',
            'LP_matching' or 'LP'
        """
        if algorithm in ['Edmonds', 'LP_matching', 'LP']:
            return True

        raise ValueError('algorithm must be set to \'Edmonds\', '
                         '\'LP_matching\' or \'LP\'')

    @doc_index('Overwritten methods')
    def is_biconnected(self):
        r"""
        Check whether the (matching covered) graph is biconnected.

        A biconnected graph is a connected graph on two or more vertices that
        is not broken into disconnected pieces by deleting any single vertex.
        By definition of matching covered graphs, it follows that a graph, that
        is matching covered, is biconnected.

        Observe that `K_2` (upto multiple edges) is biconnected. A matching
        covered graph `G`, that is not `K_2` (upto multiple edges), has at
        least four vertices and four edges. Consider any two adjacent edges
        (`e` and `f`) of `G`. Take a perfect matching `M` of `G` containing
        the edge `e` and a different perfect matching `N` of `G` containing
        the edge `f`. Observe that the symmetric difference of `M` and `N`
        has a cycle containing both of the edges `e` and `f`. Thus, each edge
        of `G` is contained in some cycle of `G`. Therefore, `G` is
        biconnected.

        .. NOTE::

            This method overwrites the
            :meth:`~sage.graphs.generic_graph.GenericGraph.is_biconnected`
            method in order to return ``True`` as matching covered graphs are
            biconnected.

        EXAMPLES:

        The complete graph on two vertices is the smallest biconnected graph
        that is matching covered::

            sage: K2 = graphs.CompleteGraph(2)
            sage: G = MatchingCoveredGraph(K2)
            sage: G.is_biconnected()
            True

        Petersen graph is matching covered and biconnected::

            sage: P = graphs.PetersenGraph()
            sage: G = MatchingCoveredGraph(P)
            sage: G.is_biconnected()
            True

        .. SEEALSO::

            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.is_connected`
        """
        return True

    @doc_index('Bricks, braces and tight cut decomposition')
    def is_brace(self, coNP_certificate=False):
        r"""
        Check if the (matching covered) graph is a brace.

        A matching covered graph which is free of nontrivial tight cuts is
        called a *brace* if it is bipartite. Let `G := (A \cup B, E)` be a
        bipartite matching covered graph on six or more vertices. The
        following statements are equivalent [LM2024]_:

        1. `G` is a brace (aka free of nontrivial tight cuts).
        2. `G - a_1 - a_2 - b_1 - b_2` has a perfect matching for any two
           distinct vertices `a_1` and `a_2` in `A` and any two distinct
           vertices `b_1` and `b_2` in `B`.
        3. `G` is two extendable (any two nonadjacent distinct edges can be
           extended to some perfect matching of `G`).
        4. `|N(X)| \geq |X| + 2`, for all `X ⊂ A` such that `0 < |X| <
           |A| - 1`, where `N(S) := \{b \mid (a, b) \in E ∧ a \in S\}` is called
           the neighboring set of `S`.
        5. `G - a - b` is matching covered, for some perfect matching `M` of
           `G` and for each edge `ab` in `M`.

        We shall be using the 5th characterization mentioned above in order
        to determine whether the provided bipartite matching covered graph
        is a brace or not using *M*-alternating tree search [LZ2001]_.

        INPUT:

        - ``coNP_certificate`` -- boolean (default: ``False``)

        OUTPUT:

        - If the input matching covered graph is not bipartite, a
          :exc:`ValueError` is returned.

        - If the input bipartite matching covered graph is a brace, a boolean
          ``True`` is returned if ``coNP_certificate`` is set to ``False``
          otherwise a 5-tuple ``(True, None, None, None, None)`` is returned.

        - If the input bipartite matching covered graph is not a brace, a
          boolean ``False`` is returned if ``coNP_certificate`` is set to
          ``False`` otherwise a 5-tuple of

          1. a boolean ``False``,

          2. a list of edges constituting a nontrivial tight cut (which is a
             nontrivial barrier cut)

          3. a set of vertices of one of the shores of the nontrivial tight cut

          4. a string 'nontrivial tight cut'

          5. a set of vertices showing the respective barrier

          is returned.

        EXAMPLES:

        The complete graph on two vertices `K_2` is the smallest brace::

            sage: K = graphs.CompleteGraph(2)
            sage: G = MatchingCoveredGraph(K)
            sage: G.is_brace()
            True

        The cycle graph on four vertices `C_4` is a brace::

            sage: C = graphs.CycleGraph(4)
            sage: G = MatchingCoveredGraph(C)
            sage: G.is_brace()
            True

        Each graph that is isomorphic to a biwheel is a brace::

            sage: B = graphs.BiwheelGraph(15)
            sage: G = MatchingCoveredGraph(B)
            sage: G.is_brace()
            True

        A circular ladder graph of order eight or more on `2n` vertices for
        an even `n` is a brace::

            sage: n = 10
            sage: CL = graphs.CircularLadderGraph(n)
            sage: G = MatchingCoveredGraph(CL)
            sage: G.is_brace()
            True

        A moebius ladder graph of order six or more on `2n` vertices for an odd
        `n` is a brace::

            sage: n = 11
            sage: ML = graphs.MoebiusLadderGraph(n)
            sage: G = MatchingCoveredGraph(ML)
            sage: G.is_brace()
            True

        Note that the union of the above mentioned four families of braces,
        that are:

        1. the biwheel graph ``BiwheelGraph(n)``,
        2. the circular ladder graph ``CircularLadderGraph(n)`` for even ``n``,
        3. the moebius ladder graph ``MoebiusLadderGraph(n)`` for odd ``n``,

        is referred to as the *McCuaig* *family* *of* *braces.*

        The only simple brace of order six is the complete graph of the same
        order, that is `K_{3, 3}`::

            sage: L = list(graphs(6,
            ....:          lambda G: G.size() <= 15 and
            ....:                    G.is_bipartite())
            ....: )
            sage: L = list(G for G in L if G.is_connected() and
            ....:                          G.is_matching_covered()
            ....: )
            sage: M = list(MatchingCoveredGraph(G) for G in L)
            sage: B = list(G for G in M if G.is_brace())
            sage: K = graphs.CompleteBipartiteGraph(3, 3)
            sage: G = MatchingCoveredGraph(K)
            sage: next(iter(B)).is_isomorphic(G)
            True

        The nonplanar `K_{3, 3}`-free brace Heawood graph is the unique cubic
        graph of girth six with the fewest number of vertices (that is 14).
        Note that by `K_{3, 3}`-free, it shows that the Heawood graph does not
        contain a subgraph that is isomophic to a graph obtained by
        bisubdivision of `K_{3, 3}`::

            sage: K = graphs.CompleteBipartiteGraph(3, 3)
            sage: J = graphs.HeawoodGraph()
            sage: H = MatchingCoveredGraph(J)
            sage: H.is_brace() and not H.is_planar() and \
            ....: H.is_regular(k=3) and H.girth() == 6
            True

        Braces of order six or more are 3-connected::

            sage: H = graphs.HexahedralGraph()
            sage: G = MatchingCoveredGraph(H)
            sage: G.is_brace() and G.is_triconnected()
            True

        Braces of order four or more are 2-extendable::

            sage: H = graphs.EllinghamHorton54Graph()
            sage: G = MatchingCoveredGraph(H)
            sage: G.is_brace()
            True
            sage: e = next(G.edge_iterator(labels=False)); f = None
            sage: for f in G.edge_iterator(labels=False):
            ....:     if not (set(e) & set(f)):
            ....:          break
            sage: S = [u for x in [e, f] for u in set(x)]
            sage: J = H.copy(); J.delete_vertices(S)
            sage: M = Graph(J.matching())
            sage: M.add_edges([e, f])
            sage: if all(d == 1 for d in M.degree()) and \
            ....:    G.order() == M.order() and \
            ....:    G.order() == 2*M.size():
            ....:      print(f'graph {G} is 2-extendable')
            graph Ellingham-Horton 54-graph is 2-extendable

        Every edge in a brace of order at least six is removable::

            sage: H = graphs.CircularLadderGraph(8)
            sage: G = MatchingCoveredGraph(H)
            sage: # len(G.removble_edges()) == G.size()
            # True

        Every brace of order eight has the hexahedral graph as a spanning
        subgraph::

            sage: H = graphs.HexahedralGraph()
            sage: L = list(graphs(8,
            ....:          lambda G: G.size() <= 28 and
            ....:                    G.is_bipartite())
            ....: )
            sage: L = list(G for G in L if G.is_connected() and
            ....:                          G.is_matching_covered()
            ....: )
            sage: M = list(MatchingCoveredGraph(G) for G in L)
            sage: B = list(G for G in M if G.is_brace())
            sage: C = list(G for G in M if Graph(G).subgraph_search(H) is not None)
            sage: B == C
            True

        For every brace `G[A, B]` of order at least six, the graph
        `G - a_1 - a_2 - b_1 - b_2` has a perfect matching for any two distinct
        vertices `a_1` and `a_2` in `A` and any two distinct vertices `b_1` and
        `b_2` in `B`::

            sage: H =  graphs.CompleteBipartiteGraph(10, 10)
            sage: G = MatchingCoveredGraph(H)
            sage: G.is_brace()
            True
            sage: S = [0, 1, 10, 12]
            sage: G.delete_vertices(S)
            sage: G.has_perfect_matching()
            True

        For a brace `G[A, B]` of order six or more, `|N(X)| \geq |X| + 2`, for
        all `X \subset A` such that `0 < |X| <|A| - 1`, where
        `N(S) := \{b | (a, b) \in E \wedge a \in S\}` is called the neighboring set
        of `S`::

            sage: H = graphs.MoebiusLadderGraph(15)
            sage: G = MatchingCoveredGraph(H)
            sage: G.is_brace()
            True
            sage: A, _ = G.bipartite_sets()
            sage: # needs random
            sage: X = random.sample(list(A), random.randint(1, len(A) - 1))
            sage: N = {v for u in X for v in G.neighbor_iterator(u)}
            sage: len(N) >= len(X) + 2
            True

        For a brace `G` of order four or more with a perfect matching `M`, the
        graph `G - a - b` is matching covered for each edge `(a, b)` in `M`::

            sage: H = graphs.HeawoodGraph()
            sage: G = MatchingCoveredGraph(H)
            sage: G.is_brace()
            True
            sage: M = G.get_matching()
            sage: L = []
            sage: for a, b, *_ in M:
            ....:     J = G.copy(); J.delete_vertices([a, b])
            ....:     if J.is_matching_covered():
            ....:          L.append(J)
            sage: len(L) == len(M)
            True

        A cycle graph of order six or more is a bipartite matching covered
        graph, but is not a brace::

            sage: C = graphs.CycleGraph(10)
            sage: G = MatchingCoveredGraph(C)
            sage: G.is_brace()
            False

        A ladder graph of order six or more is a bipartite matching covered
        graph, that is not a brace. The tight cut decomposition of a ladder
        graph produces a list graphs the underlying graph of each of which
        is isomorphic to a 4-cycle::

            sage: L = graphs.LadderGraph(10)
            sage: G = MatchingCoveredGraph(L)
            sage: G.is_brace()
            False

        One may set the ``coNP_certificate`` to be ``True``::

            sage: H = graphs.HexahedralGraph()
            sage: G = MatchingCoveredGraph(H)
            sage: G.is_brace(coNP_certificate=True)
            (True, None, None, None, None)
            sage: C = graphs.CycleGraph(6)
            sage: D = MatchingCoveredGraph(C)
            sage: is_brace, nontrivial_tight_cut, nontrivial_odd_component, \
            ....: nontrivial_tight_cut_variant, cut_identifier = \
            ....: D.is_brace(coNP_certificate=True)
            sage: is_brace is False
            True
            sage: J = C.subgraph(vertices=nontrivial_odd_component)
            sage: J.is_isomorphic(graphs.PathGraph(3))
            True
            sage: len(nontrivial_tight_cut) == 2
            True
            sage: nontrivial_tight_cut_variant
            'nontrivial barrier cut'
            sage: # Corresponding barrier
            sage: cut_identifier == {a for u, v, *_ in nontrivial_tight_cut for a in [u, v] \
            ....: if a not in nontrivial_odd_component}
            True
            sage: for u, v, *_ in nontrivial_tight_cut:
            ....:     assert (u in nontrivial_odd_component and v not in nontrivial_odd_component)
            sage: L = graphs.LadderGraph(3) # A ladder graph with two constituent braces
            sage: G = MatchingCoveredGraph(L)
            sage: is_brace, nontrivial_tight_cut, nontrivial_odd_component, cut_variant, cut_identifier = \
            ....: G.is_brace(coNP_certificate=True)
            sage: is_brace is False
            True
            sage: G1 = L.copy()
            sage: G1.merge_vertices(list(nontrivial_odd_component))
            sage: G1.to_simple().is_isomorphic(graphs.CycleGraph(4))
            True
            sage: G2 = L.copy()
            sage: G2.merge_vertices([v for v in G if v not in nontrivial_odd_component])
            sage: G2.to_simple().is_isomorphic(graphs.CycleGraph(4))
            True
            sage: cut_variant
            'nontrivial barrier cut'
            sage: cut_identifier == {a for u, v, *_ in nontrivial_tight_cut for a in [u, v] \
            ....: if a not in nontrivial_odd_component}
            True
            sage: H = graphs.CompleteBipartiteGraph(3, 3)
            sage: H.delete_edge(0, 3)
            sage: G = MatchingCoveredGraph(H)
            sage: G.is_brace(coNP_certificate=True)
            (False,
             [(4, 1, None), (5, 1, None), (4, 2, None), (5, 2, None)],
             {0, 4, 5},
             'nontrivial barrier cut',
             {1, 2})

        If the input matching covered graph is nonbipartite, a
        :exc:`ValueError` is thrown::

            sage: K4 = graphs.CompleteGraph(4)
            sage: G = MatchingCoveredGraph(K4)
            sage: G.is_brace()
            Traceback (most recent call last):
            ...
            ValueError: the input graph is not bipartite
            sage: P = graphs.PetersenGraph()
            sage: H = MatchingCoveredGraph(P)
            sage: H.is_brace(coNP_certificate=True)
            Traceback (most recent call last):
            ...
            ValueError: the input graph is not bipartite

        .. SEEALSO::

            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.is_brick`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.bricks_and_braces`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.number_of_braces`
        """
        if not self.is_bipartite():
            raise ValueError('the input graph is not bipartite')

        if self.order() < 6:
            return (True, None, None, None, None) if coNP_certificate else True

        A, B = self.bipartite_sets()
        matching = set(self.get_matching())
        matching_neighbor = {x: y for u, v, *_ in matching for x, y in [(u, v), (v, u)]}

        for e in matching:
            u, v, *_ = e

            # Let G denote the undirected graph self, and
            # let the graph H(e) := G — u — v
            H = Graph(self, multiedges=False)
            H.delete_vertices([u, v])

            if not H.is_connected() or not H.is_matching_covered(list(matching - set([e]))):
                if not coNP_certificate:
                    return False

                # Construct the digraph D(e)(A ∪ B, F) defined as follows:
                from sage.graphs.digraph import DiGraph
                D = DiGraph()

                # For each edge (a, b) in E(H(e)) ∩ M with a in A, b —> a in D(e).
                # For each edge (a, b) in E(H(e)) with a in A, a —> b in D(e).
                for a, b in H.edge_iterator(labels=False, sort_vertices=True):

                    if a in B:
                        a, b = b, a

                    D.add_edge((a, b))
                    if matching_neighbor[a] == b:
                        D.add_edge((b, a))

                # H(e) is matching covered iff D(e) is strongly connected.
                # Check if D(e) is strongly connected using Kosaraju's algorithm
                def dfs(x, visited, neighbor_iterator):
                    stack = [x]  # a stack of xertices

                    while stack:
                        x = stack.pop()
                        visited.add(x)

                        for y in neighbor_iterator(x):
                            if y not in visited:
                                stack.append(y)

                root = next(D.vertex_iterator())

                visited_in = set()
                dfs(root, visited_in, D.neighbor_in_iterator)

                # Since D(e) is not strongly connected, it has a directed cut T(e).
                # Note that by definition of D(e), it follows that T(e) ⊆ E(H(e)) — M.
                # Thus, T(e) is a cut of H(e), which has a shore X such that every edge of T(e) is
                # incident with a vertex in X ∩ B.

                # Moreover, M — e is a perfect matching of H(e), and thus, |X ∩ A| = |X ∩ B|
                # Consequently, Y := X + v is a shore of a nontrivial tight cut T of G

                if len(visited_in) != D.order():
                    X = visited_in
                else:
                    X = set()
                    dfs(root, X, D.neighbor_out_iterator)

                color_class = None

                for a, b in H.edge_iterator(labels=False, sort_vertices=True):
                    if (a in X) ^ (b in X):
                        x = a if a in A else b
                        color_class = x not in X
                        break

                # Obtain the color class Z ∈ {A, B} such that X ∩ Z is a vertex cover for T(e)
                # Thus, obtain Y := X + v
                X.add(u if (not color_class and u in A) or (color_class and u in B) or (color_class is None) else v)

                # Compute the nontrivial tight cut C := ∂(Y)
                C = [(x, y, w) if x in X else (y, x, w)
                    for x, y, w in self.edge_iterator(sort_vertices=True)
                    if (x in X) ^ (y in X)]

                # Obtain the barrier Z
                Z = None

                if (u in X and u in A) or (v in X and v in A):
                    Z = {b for b in B if b not in X}
                else:
                    Z = {a for a in A if a not in X}

                return (False, C, set(X), 'nontrivial barrier cut', Z)

        return (True, None, None, None, None) if coNP_certificate else True

    @doc_index('Bricks, braces and tight cut decomposition')
    def is_brick(self, coNP_certificate=False):
        r"""
        Check if the (matching covered) graph is a brick.

        A matching covered graph which is free of nontrivial tight cuts is
        called a *brick* if it is nonbipartite. A nonbipartite matching covered
        graph is a brick if and only if it is 3-connected and bicritical
        [LM2024]_.

        INPUT:

        - ``coNP_certificate`` -- boolean (default: ``False``)

        OUTPUT:

        - If the input matching covered graph is bipartite, a :exc:`ValueError`
          is returned.

        - If the input nonbipartite matching covered graph is a brick, a
          boolean ``True`` is returned if ``coNP_certificate`` is set to
          ``False``, otherwise a 5-tuple ``(True, None, None, None, None)`` is
          returned.

        - If the input nonbipartite matching covered graph is not a brick, a
          boolean ``False`` is returned if ``coNP_certificate`` is set to
          ``False``.

        - If ``coNP_certificate`` is set to ``True`` and the input nonbipartite
          graph is not a brick, a 5-tuple of

          1. a boolean ``False``,

          2. a list of lists of edges, each list constituting a nontrivial
             tight cut collectively representing a laminar tight cut,

          3. a list of set of vertices of one of the shores of those respective
             nontrivial tight cuts:

             #. In case of nontrivial barrier cuts, each of the shores is a
                nontrivial odd component with respect to a nontrivial barrier,
                thus the returned list forms mutually exclusive collection of
                (odd) sets.

             #. Otherwise each of the nontrivial tight cuts is a 2-separation
                cut, each of the shores form a subset sequence, with the
                `i` th shore being a proper subset of the `i + 1` th shore.

          4. a string showing whether the nontrivial tight cuts are barrier
             cuts (if the string is 'nontrivial barrier cut'), or 2-separation
             cuts (if the string is 'nontrivial 2-separation cut')

          5. a set of vertices showing the respective barrier if the
             nontrivial tight cuts are barrier cuts, or otherwise
             a set of two vertices constituting the corresponding
             two vertex cut (in this case the nontrivial tight cuts are
             2-separation cuts)

          is returned.

        EXAMPLES:

        The complete graph on four vertices `K_4` is the smallest brick::

            sage: K = graphs.CompleteGraph(4)
            sage: G = MatchingCoveredGraph(K)
            sage: G.is_brick()
            True

        The triangular circular ladder (a graph on six vertices), aka
        `\overline{C_6}` is a brick::

            sage: C6Bar = graphs.CircularLadderGraph(3)
            sage: G = MatchingCoveredGraph(C6Bar)
            sage: G.is_brick()
            True

        Each of Petersen graph, Bicorn graph, Tricorn graph, Cubeplex graph,
        Twinplex graph, Wagner graph is a brick::

            sage: MatchingCoveredGraph(graphs.PetersenGraph()).is_brick() and \
            ....: MatchingCoveredGraph(graphs.StaircaseGraph(4)).is_brick() and \
            ....: MatchingCoveredGraph(graphs.TricornGraph()).is_brick() and \
            ....: MatchingCoveredGraph(graphs.CubeplexGraph()).is_brick() and \
            ....: MatchingCoveredGraph(graphs.TwinplexGraph()).is_brick() and \
            ....: MatchingCoveredGraph(graphs.WagnerGraph()).is_brick()
            True

        The Murty graph is the smallest simple brick that is not odd-intercyclic::

            sage: M = graphs.MurtyGraph()
            sage: G = MatchingCoveredGraph(M)
            sage: G.is_brick()
            True

        A circular ladder graph of order six or more on `2n` vertices for an
        odd `n` is a brick::

            sage: n = 11
            sage: CL = graphs.CircularLadderGraph(n)
            sage: G = MatchingCoveredGraph(CL)
            sage: G.is_brick()
            True

        A moebius ladder graph of order eight or more on `2n` vertices for an
        even `n` is a brick::

            sage: n = 10
            sage: ML = graphs.MoebiusLadderGraph(n)
            sage: G = MatchingCoveredGraph(ML)
            sage: G.is_brick()
            True

        A wheel graph of an even order is a brick::

            sage: W = graphs.WheelGraph(10)
            sage: G = MatchingCoveredGraph(W)
            sage: G.is_brick()
            True

        A graph that is isomorphic to a truncated biwheel graph is a brick::

            sage: TB = graphs.TruncatedBiwheelGraph(15)
            sage: G = MatchingCoveredGraph(TB)
            sage: G.is_brick()
            True

        Each of the graphs in the staircase graph family with order eight or
        more is a brick::

            sage: ST = graphs.StaircaseGraph(9)
            sage: G = MatchingCoveredGraph(ST)
            sage: G.is_brick()
            True

        Bricks are 3-connected::

            sage: P = graphs.PetersenGraph()
            sage: G = MatchingCoveredGraph(P)
            sage: G.is_brick()
            True
            sage: G.is_triconnected()
            True

        Bricks are bicritical::

            sage: P = graphs.PetersenGraph()
            sage: G = MatchingCoveredGraph(P)
            sage: G.is_brick()
            True
            sage: G.is_bicritical()
            True

        Examples of nonbipartite matching covered graphs that are not
        bricks::

            sage: H = Graph([
            ....:     (0, 3), (0, 4), (0, 7),
            ....:     (1, 3), (1, 5), (1, 7),
            ....:     (2, 3), (2, 6), (2, 7),
            ....:     (4, 5), (4, 6), (5, 6)
            ....: ])
            sage: G = MatchingCoveredGraph(H)
            sage: G.is_bipartite()
            False
            sage: G.is_bicritical()
            False
            sage: G.is_triconnected()
            True
            sage: G.is_brick()
            False
            sage: H = Graph([
            ....:     (0, 1), (0, 2), (0, 3), (0, 4), (1, 2),
            ....:     (1, 5), (2, 5), (3, 4), (3, 5), (4, 5)
            ....: ])
            sage: G = MatchingCoveredGraph(H)
            sage: G.is_bipartite()
            False
            sage: G.is_bicritical()
            True
            sage: G.is_triconnected()
            False
            sage: G.is_brick()
            False

        One may set the ``coNP_certificate`` to be ``True``::

            sage: K4 = graphs.CompleteGraph(4)
            sage: G = MatchingCoveredGraph(K4)
            sage: G.is_brick(coNP_certificate=True)
            (True, None, None, None, None)
            sage: # K(4) ⊙ K(3, 3) is nonbipartite but not a brick
            sage: H = graphs.MurtyGraph(); H.delete_edge(0, 1)
            sage: G = MatchingCoveredGraph(H)
            sage: G.is_brick(coNP_certificate=True)
            (False, [[(5, 2, None), (6, 3, None), (7, 4, None)]], [{5, 6, 7}],
             'nontrivial barrier cut', {2, 3, 4})
            sage: H = Graph([
            ....:     (0, 12), (0, 13), (0, 15), (1, 4), (1, 13), (1, 14),
            ....:     (1, 19), (2, 4), (2, 13), (2, 14), (2, 17), (3, 9),
            ....:     (3, 13), (3, 16), (3, 21), (4, 6), (4, 7), (5, 7),
            ....:     (5, 8), (5, 12), (6, 8), (6, 11), (7, 10), (8, 9),
            ....:     (9, 10), (10, 11), (11, 12), (14, 15), (14, 16), (15, 16),
            ....:     (17, 18), (17, 21), (18, 19), (18, 20), (19, 20), (20, 21)
            ....: ])
            sage: G = MatchingCoveredGraph(H)
            sage: G.is_brick(coNP_certificate=True)
            (False,
             [[(12, 0, None), (4, 1, None), (4, 2, None), (9, 3, None)],
              [(19, 1, None), (17, 2, None), (21, 3, None)],
              [(15, 0, None), (14, 1, None), (14, 2, None), (16, 3, None)]],
             [{4, 5, 6, 7, 8, 9, 10, 11, 12}, {17, 18, 19, 20, 21}, {14, 15, 16}],
             'nontrivial barrier cut', {0, 1, 2, 3})
            sage: J = Graph([
            ....:     (0, 1), (0, 2), (0, 3), (0, 4), (0, 5),
            ....:     (0, 6), (0, 7), (0, 8), (0, 9), (0, 10),
            ....:     (1, 2), (1, 11), (2, 11), (3, 4), (3, 11),
            ....:     (4, 11), (5, 6), (5, 11), (6, 11), (7, 8),
            ....:     (7, 11), (8, 11), (9, 10), (9, 11), (10, 11)
            ....: ])
            sage: G = MatchingCoveredGraph(J)
            sage: G.is_brick(coNP_certificate=True)
            (False,
             [[(0, 3, None),
               (0, 4, None),
               (0, 5, None),
               (0, 6, None),
               (0, 7, None),
               (0, 8, None),
               (0, 9, None),
               (0, 10, None),
               (1, 11, None),
               (2, 11, None)],
              [(0, 5, None),
               (0, 6, None),
               (0, 7, None),
               (0, 8, None),
               (0, 9, None),
               (0, 10, None),
               (1, 11, None),
               (2, 11, None),
               (3, 11, None),
               (4, 11, None)],
              [(0, 7, None),
               (0, 8, None),
               (0, 9, None),
               (0, 10, None),
               (1, 11, None),
               (2, 11, None),
               (3, 11, None),
               (4, 11, None),
               (5, 11, None),
               (6, 11, None)],
              [(0, 9, None),
               (0, 10, None),
               (1, 11, None),
               (2, 11, None),
               (3, 11, None),
               (4, 11, None),
               (5, 11, None),
               (6, 11, None),
               (7, 11, None),
               (8, 11, None)]],
             [{0, 1, 2},
              {0, 1, 2, 3, 4},
              {0, 1, 2, 3, 4, 5, 6},
              {0, 1, 2, 3, 4, 5, 6, 7, 8}],
             'nontrivial 2-separation cut',
             {0, 11})

        If the input matching covered graph is bipartite, a
        :exc:`ValueError` is thrown::

            sage: H = graphs.HexahedralGraph()
            sage: G = MatchingCoveredGraph(H)
            sage: G.is_brick()
            Traceback (most recent call last):
            ...
            ValueError: the input graph is bipartite
            sage: J = graphs.HeawoodGraph()
            sage: G = MatchingCoveredGraph(J)
            sage: G.is_brick(coNP_certificate=True)
            Traceback (most recent call last):
            ...
            ValueError: the input graph is bipartite

        .. SEEALSO::

            - :meth:`~sage.graphs.graph.Graph.is_bicritical`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.is_brace`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.bricks_and_braces`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.number_of_bricks`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.number_of_petersen_bricks`
        """
        if self.is_bipartite():
            raise ValueError('the input graph is bipartite')

        # Check if G is bicritical
        bicritical, certificate = self.is_bicritical(coNP_certificate=True)

        if not bicritical:
            if not coNP_certificate:
                return False

            # G has a pair of vertices u, v such that G - u - v is not matching
            # covered, thus has a nontrivial barrier B containing both u and v.
            u, _ = certificate
            B = self.maximal_barrier(u)

            H = Graph(self)
            H.delete_vertices(B)

            # Let K be a nontrivial odd component of H := G - B. Note that
            # there exists at least one such K since G is nonbipartite
            nontrivial_odd_components = [
                set(component) for component in H.connected_components(sort=True)
                if len(component) % 2 and len(component) > 1
            ]

            # Find a laminar set of nontrivial barrier cuts
            C = [[(u, v, w) if u in nontrivial_odd_component else (v, u, w)
                  for u, v, w in self.edge_iterator()
                  if (u in nontrivial_odd_component) ^ (v in nontrivial_odd_component)]
                 for nontrivial_odd_component in nontrivial_odd_components]

            return (False, C, nontrivial_odd_components, 'nontrivial barrier cut', B)

        # Check if G is 3-connected
        if self.is_triconnected():
            return (True, None, None, None, None) if coNP_certificate else True

        # G has a 2-vertex cut
        # Compute the SPQR-tree decomposition
        spqr_tree = self.spqr_tree()
        two_vertex_cut = []

        # Check for 2-vertex cuts in a P node
        # Since the graph is matching covered, it is free of cut vertices
        # It can be shown using counting arguments that the spqr tree
        # decomposition for a bicritical graph, that is 2-connected but not
        # 3-connected, is free of 'S' nodes
        for u in spqr_tree:
            if u[0] == 'P':
                two_vertex_cut.extend(u[1])
                break

        # If no 2-vertex cut found, look for R nodes
        if not two_vertex_cut:
            from collections import Counter
            R_frequency = Counter()

            for t, g in spqr_tree:
                if t == 'R':
                    R_frequency.update(g)

            # R frequency must be at least 2,
            # since the graph is 2-connected but not 3-connected
            two_vertex_cut = [u for u, f in R_frequency.items() if f >= 2][:2]

        # We obtain a 2-vertex cut (u, v)
        H = Graph(self)
        H.delete_vertices(two_vertex_cut)

        # Check if all components of H are odd
        components = H.connected_components(sort=True)

        # Find a nontrivial odd component
        nontrivial_tight_cut_variation = 'nontrivial 2-separation cut'
        nontrivial_odd_components = []

        for index, component in enumerate(components):
            if index == len(components) - 1:
                continue
            elif not index:
                nontrivial_odd_components.append(set(components[0] + [two_vertex_cut[0]]))
            else:
                nontrivial_odd_component = nontrivial_odd_components[-1].copy()
                nontrivial_odd_component.update(component)
                nontrivial_odd_components.append(nontrivial_odd_component)

        C = [[(u, v, w) if u in nontrivial_odd_component else (v, u, w)
              for u, v, w in self.edge_iterator()
              if (u in nontrivial_odd_component) ^ (v in nontrivial_odd_component)]
             for nontrivial_odd_component in nontrivial_odd_components]

        # Edge (u, v, w) in C are formatted so that u is in a nontrivial odd component
        return (False, C, nontrivial_odd_components, nontrivial_tight_cut_variation, set(two_vertex_cut)) if coNP_certificate else False

    @doc_index('Overwritten methods')
    def loop_edges(self, labels=True):
        r"""
        Return a list of all loops in the (matching covered) graph.

        .. NOTE::

            This method overwrites the
            :meth:`~sage.graphs.generic_graph.GenericGraph.loop_edges` method
            in order to return an empty list as matching covered graphs are
            free of looped edges.

        INPUT:

        - ``labels`` -- boolean (default: ``True``); whether returned edges
          have labels (``(u,v,l)``) or not (``(u,v)``).

        OUTPUT:

        - A list capturing the edges that are loops in the matching covered
          graph; note that, the list is empty since matching covered graphs do
          not contain any looped edges.

        EXAMPLES:

        A matching covered graph, for instance the Heawood graph, by
        definition, is always free of loops::

            sage: H = graphs.HeawoodGraph()
            sage: G = MatchingCoveredGraph(H)
            sage: G
            Matching covered heawood graph: graph on 14 vertices
            sage: G.add_edge(0, 0)
            Traceback (most recent call last):
            ...
            ValueError: loops are not allowed in matching covered graphs
            sage: G.loops()
            []
            sage: G.loop_edges()
            []

        A matching covered graph may support multiple edges, still no
        loops are allowed::

            sage: C = graphs.CycleGraph(4)
            sage: G = MatchingCoveredGraph(C)
            sage: G.allow_multiple_edges(True)
            sage: G
            Matching covered cycle graph: multi-graph on 4 vertices
            sage: G.add_edge(0, 1, 'label')
            sage: G.add_edge(0, 0)
            Traceback (most recent call last):
            ...
            ValueError: loops are not allowed in matching covered graphs
            sage: G.edges(sort=False)
            [(0, 1, None), (0, 1, 'label'), (0, 3, None), (1, 2, None), (2, 3, None)]
            sage: G.loops()
            []
            sage: G.loop_edges()
            []

        One may set the ``label`` to either ``True`` or ``False``::

            sage: G.loop_edges(labels=False)
            []
            sage: G.loops(labels=True)
            []

        .. SEEALSO::

            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.allow_loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.allows_loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.has_loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.loop_vertices`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.number_of_loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.remove_loops`
        """
        return []

    @doc_index('Overwritten methods')
    def loop_vertices(self):
        r"""
        Return a list of vertices with loops.

        .. NOTE::

            This method overwrites the
            :meth:`~sage.graphs.generic_graph.GenericGraph.loop_vertices`
            method in order to return an empty list as matching covered graphs
            are free of vertices that have looped edges.

        OUTPUT:

        - A list capturing the vertices that have loops in the matching covered
          graph; note that, the list is empty since matching covered graphs do
          not contain any looped edges.

        EXAMPLES:

        A matching covered graph, for instance the Möbius graph of order 8, by
        definition, is always free of loops::

            sage: M = graphs.MoebiusLadderGraph(4)
            sage: G = MatchingCoveredGraph(M)
            sage: G
            Matching covered moebius ladder graph: graph on 8 vertices
            sage: G.add_edge(0, 0)
            Traceback (most recent call last):
            ...
            ValueError: loops are not allowed in matching covered graphs
            sage: G.loop_vertices()
            []

        A matching covered graph may support multiple edges, still no
        loops are allowed::

            sage: S = graphs.StaircaseGraph(4)
            sage: G = MatchingCoveredGraph(S)
            sage: G.allow_multiple_edges(True)
            sage: G
            Matching covered staircase graph: multi-graph on 8 vertices
            sage: G.add_edge(0, 1, 'label')
            sage: G.add_edge(0, 0)
            Traceback (most recent call last):
            ...
            ValueError: loops are not allowed in matching covered graphs
            sage: G.edges(sort=False)
            [(0, 1, None), (0, 1, 'label'), (0, 3, None), (0, 6, None),
             (1, 2, None), (1, 4, None), (2, 5, None), (2, 7, None),
             (3, 4, None), (3, 6, None), (4, 5, None), (5, 7, None),
             (6, 7, None)]
            sage: G.loop_vertices()
            []

        .. SEEALSO::

            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.allow_loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.allows_loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.has_loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.loop_edges`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.number_of_loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.remove_loops`
        """
        return []

    loops = loop_edges

    @doc_index('Overwritten methods')
    def number_of_loops(self):
        r"""
        Return the number of edges that are loops.

        .. NOTE::

            This method overwrites the
            :meth:`~sage.graphs.generic_graph.GenericGraph.number_of_loops`
            method in order to return 0 as matching covered graphs are free
            of looped edges.

        OUTPUT:

        - An integer, 0 is returned, since matching covered graphs do not
          contain zero loops.

        EXAMPLES:

        A matching covered graph, for instance the Truncated biwheel graph,
        by definition, is always free of loops::

            sage: T = graphs.TruncatedBiwheelGraph(5)
            sage: G = MatchingCoveredGraph(T)
            sage: G
            Matching covered truncated biwheel graph: graph on 10 vertices
            sage: G.add_edge(0, 0)
            Traceback (most recent call last):
            ...
            ValueError: loops are not allowed in matching covered graphs
            sage: G.loop_vertices()
            []
            sage: G.number_of_loops()
            0

        A matching covered graph may support multiple edges, still no
        loops are allowed::

            sage: B = graphs.BiwheelGraph(4)
            sage: G = MatchingCoveredGraph(B)
            sage: G.allow_multiple_edges(True)
            sage: G
            Matching covered biwheel graph: multi-graph on 8 vertices
            sage: G.add_edge(0, 1, 'label')
            sage: G.add_edge(0, 0)
            Traceback (most recent call last):
            ...
            ValueError: loops are not allowed in matching covered graphs
            sage: G.edges(sort=False)
            [(0, 1, None), (0, 1, 'label'), (0, 5, None), (0, 7, None),
             (1, 2, None), (1, 6, None), (2, 3, None), (2, 7, None),
             (3, 4, None), (3, 6, None), (4, 5, None), (4, 7, None),
             (5, 6, None)]
            sage: G.loop_vertices()
            []
            sage: G.number_of_loops()
            0

        .. SEEALSO::

            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.allow_loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.allows_loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.has_loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.loop_edges`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.loop_vertices`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.remove_loops`
        """
        return 0

    @doc_index('Overwritten methods')
    def remove_loops(self, vertices=None):
        r"""
        Remove loops on vertices in ``vertices``.

        .. NOTE::

            This method overwrites the
            :meth:`~sage.graphs.generic_graph.GenericGraph.remove_loops` method
            in order to return without any alteration as matching covered
            graphs are free of looped edges.

        INPUT:

        - ``vertices`` -- (default: ``None``) iterator container of vertex
          labels corresponding to which the looped edges are to be removed. If
          ``vertices`` is ``None``, remove all loops.

        OUTPUT:

        - Nothing is returned, as a matching covered graph is already devoid of
          any loops.

        EXAMPLES:

        A matching covered graph, for instance the Wheel graph of order six, is
        always free of loops::

            sage: W = graphs.WheelGraph(6)
            sage: G = MatchingCoveredGraph(W)
            sage: G
            Matching covered wheel graph: graph on 6 vertices
            sage: G.add_edge(0, 0)
            Traceback (most recent call last):
            ...
            ValueError: loops are not allowed in matching covered graphs
            sage: G.remove_loops()
            sage: G.edges(sort=True)
            [(0, 1, None), (0, 2, None), (0, 3, None), (0, 4, None),
             (0, 5, None), (1, 2, None), (1, 5, None), (2, 3, None),
             (3, 4, None), (4, 5, None)]

        A matching covered graph may support multiple edges, still no
        loops are allowed::

            sage: K = graphs.CompleteGraph(2)
            sage: G = MatchingCoveredGraph(K)
            sage: G.allow_multiple_edges(True)
            sage: G
            Matching covered complete graph: multi-graph on 2 vertices
            sage: G.add_edge(0, 1, 'label')
            sage: G.add_edge(0, 0)
            Traceback (most recent call last):
            ...
            ValueError: loops are not allowed in matching covered graphs
            sage: G.edges(sort=False)
            [(0, 1, None), (0, 1, 'label')]
            sage: G.remove_loops(vertices=[0, 1])
            sage: G.edges(sort=False)
            [(0, 1, None), (0, 1, 'label')]
            sage: G.remove_loops(vertices=[0..100])

        Note that the parameter ``vertices`` must be either ``None`` or an
        iterable::

            sage: G.remove_loops(vertices='')
            sage: G.edges(sort=False)
            [(0, 1, None), (0, 1, 'label')]
            sage: G.remove_loops(vertices=None)
            sage: G.edges(sort=False)
            [(0, 1, None), (0, 1, 'label')]
            sage: G.remove_loops(vertices=0)
            Traceback (most recent call last):
            ...
            TypeError: 'Integer' object is not iterable
            sage: G.remove_loops(vertices=False)
            Traceback (most recent call last):
            ...
            TypeError: 'bool' object is not iterable

        .. SEEALSO::

            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.allow_loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.allows_loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.has_loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.loop_edges`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.loop_vertices`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.loops`
            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.number_of_loops`
        """
        from collections.abc import Iterable

        if vertices is not None and not isinstance(vertices, Iterable):
            raise TypeError(f'\'{vertices.__class__.__name__}\' '
                            'object is not iterable')

    @doc_index('Overwritten methods')
    def subdivide_edge(self, *args):
        r"""
        Subdivide an edge `k` times.

        .. NOTE::

            This method overwrites the
            :meth:`~sage.graphs.generic_graph.GenericGraph.subdivide_edge`
            method to ensure that resultant graph is also matching covered.

        INPUT:

        The following forms are all accepted to subdivide `8` times the edge
        between vertices `1` and `2` labeled with ``'label'``.

        - ``G.subdivide_edge(1, 2, 8 )``
        - ``G.subdivide_edge((1, 2), 8 )``
        - ``G.subdivide_edge(1, 2, 'label', 8 )``
        - ``G.subdivide_edge((1, 2, 'label'), 8 )``

        .. NOTE::

            * If the given edge is labelled with `l`, all the edges created by
              the subdivision will have the same label.

            * If no label given, the label obtained from
              :meth:`~sage.graphs.generic_graph.GenericGraph.edge_label` will
              be used. If multiple such labels are present, the first one in
              the returned list will be used.

            * The number of subdivisions must be a nonnegative even integer in
              order to ensure the resultant graph is matching covered.

            * In the context of matching covered graphs, *bisubdividing* an edge
              *t* times is defined as subdividing the edge *2t* times, for some
              nonnegative integer *t*.

            * For two input arguments, the first one must be of the form
              `(u, v)` or `(u, v, label)`

        OUTPUT:

        - If an existent edge is provided with a valid format and the parameter
          ``k`` in the argument is a nonnegative even integer then the graph is
          updated and nothing is returned, otherwise a :exc:`ValueError` is
          returned if ``k`` is not a nonnegative even integer,

        - If the graph does not contain the edge provided, a :exc:`ValueError`
          is returned. Also, a :exc:`ValueError` is thrown in case the provided
          arguments are not valid.

        EXAMPLES:

        Subdividing `4` times an edge of the Petersen graph. Please note that
        the perfect matching captured at ``self.get_matching()`` also gets
        updated::

            sage: P = graphs.PetersenGraph()
            sage: G = MatchingCoveredGraph(P)
            sage: from collections import Counter
            sage: V, E = set(G.vertices()), list(G.edges(sort=True, sort_vertices=True))
            sage: M = set(G.get_matching())
            sage: G.subdivide_edge(0, 1, 4)
            sage: W, F = set(G.vertices()), list(G.edges(sort=True, sort_vertices=True))
            sage: N = set(G.get_matching())
            sage: sorted(W - V)
            [10, 11, 12, 13]
            sage: sorted(list((Counter(F) - Counter(E)).elements())), \
            ....: sorted(list((Counter(E) - Counter(F)).elements()))
            ([(0, 10, None), (1, 13, None), (10, 11, None), (11, 12, None),
             (12, 13, None)], [(0, 1, None)])
            sage: if (0, 1, None) in M:
            ....:     assert sorted(N - M), sorted(M - N) == \
            ....:         ([(0, 10, None), (1, 13, None), (11, 12, None)], [(0, 1, None)])
            ....: else:
            ....:     assert sorted(N - M), sorted(M - N) == \
            ....:         ([(10, 11, None), (12, 13, None)], [])

        Subdividing a multiple edge/ some multiple edges::

            sage: K = graphs.CycleGraph(4)
            sage: K.allow_multiple_edges(1)
            sage: K.add_edges([(0, 1, 2), (0, 1, 3), (0, 1, 0.5)])
            sage: K.delete_edge(0, 1, None)
            sage: G = MatchingCoveredGraph(K)
            sage: from collections import Counter
            sage: V, E = set(G.vertices()), list(G.edges(sort=True, sort_vertices=True))
            sage: G.edge_label(0, 1)
            [2, 3, 0.500000000000000]
            sage: G.subdivide_edge((0, 1), 6)  # the edge: (0, 1, 2)
            sage: G.subdivide_edge((0, 1, 3), 2)  # the edge: (0, 1, 3)
            sage: G.subdivide_edge(1, 2, None, 2)  # the edge: (1, 2, None)
            sage: W, F = set(G.vertices()), list(G.edges(sort=True, sort_vertices=True))
            sage: sorted(W - V)
            [4, 5, 6, 7, 8, 9, 10, 11, 12, 13]
            sage: sorted(list((Counter(F) - Counter(E)).elements())), \
            ....: sorted(list((Counter(E) - Counter(F)).elements()))
            ([(0, 4, 2), (0, 10, 3), (1, 9, 2), (1, 11, 3), (1, 12, None), (2, 13, None),
              (4, 5, 2), (5, 6, 2), (6, 7, 2), (7, 8, 2), (8, 9, 2), (10, 11, 3),
              (12, 13, None)], [(0, 1, 2), (0, 1, 3), (1, 2, None)])

        Setting ``k`` to `0` does not change the graph::

            sage: H = G.copy()
            sage: G.subdivide_edge(0, 4, 0)  # the edge: (0, 4, None)
            sage: H == G and H.edges(sort=True) == G.edges(sort=True)
            True

        If too many or too less arguments are given, an exception is raised::

            sage: G.subdivide_edge(0, 1, 'label', 4, 6)
            Traceback (most recent call last):
            ...
            ValueError: this method takes at least 2 and at most 4 arguments
            sage: G.subdivide_edge((0, 1, 'label', 4))
            Traceback (most recent call last):
            ...
            ValueError: this method takes at least 2 and at most 4 arguments

        A :exc:`ValueError` is returned for `k` being in an invalid format or
        being not an even nonnegative, or for nonexistent or invalid edges::

            sage: G.subdivide_edge(0, 4, 'label')  # No. of subdivision: 'label'
            Traceback (most recent call last):
            ...
            TypeError: '<' not supported between instances of 'str' and 'int'
            sage: G.subdivide_edge(0, 4, 3)  # No. of subdivisions: 3
            Traceback (most recent call last):
            ...
            ValueError: the number of subdivisions must be a nonnegative even integer,
            but found 3
            sage: G.subdivide_edge(0, 4, -1)  # No. of subdivisions: -1
            Traceback (most recent call last):
            ...
            ValueError: the number of subdivisions must be a nonnegative even integer,
            but found -1
            sage: G.subdivide_edge(0, 4, 0.5)  # No. of subdivisions: 0.5
            Traceback (most recent call last):
            ...
            ValueError: the number of subdivisions must be a nonnegative even integer,
            but found 0.500000000000000
            sage: G.subdivide_edge(0, 5, 4)
            Traceback (most recent call last):
            ...
            ValueError: the given edge (0, 5, None) does not exist
            sage: G.subdivide_edge((0, 5), 4)
            Traceback (most recent call last):
            ...
            ValueError: the given edge (0, 5, None) does not exist
            sage: G.subdivide_edge((0, 1, 'label', None), 4)
            Traceback (most recent call last):
            ...
            ValueError: for two input arguments, the first one must be of the form
            (u, v) or (u, v, l), but found: (0, 1, 'label', None)
            sage: G.subdivide_edge((0), 4)
            Traceback (most recent call last):
            ...
            TypeError: object of type 'sage.rings.integer.Integer' has no len()

        .. SEEALSO::

            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.subdivide_edges`
        """
        u, v, l, k = None, None, None, None

        if len(args) == 2:
            edge, k = args

            if len(edge) == 2:
                u, v = edge
                if not self.has_edge(u, v):
                    raise ValueError(f'the given edge {(u, v, None)} does not exist')

                l = self.edge_label(u, v)
                if self.allows_multiple_edges() and isinstance(l, list):
                    l = l[0]

            elif len(edge) == 3:
                u, v, l = edge

            else:
                raise ValueError('for two input arguments, the first one must be '
                                 f'of the form (u, v) or (u, v, l), but found: {edge}')

        elif len(args) == 3:
            u, v, k = args

            if not self.has_edge(u, v):
                raise ValueError(f'the given edge {(u, v, None)} does not exist')

            l = self.edge_label(u, v)
            if isinstance(l, list):
                l = next(iter(l))

        elif len(args) == 4:
            u, v, l, k = args

        else:
            raise ValueError('this method takes at least 2 and at most 4 arguments')

        if not self.has_edge(u, v, l):
            raise ValueError(f'the given edge {(u, v, l)} does not exist')

        if k < 0 or k % 2:
            raise ValueError('the number of subdivisions must be a '
                             f'nonnegative even integer, but found {k}')

        if not k:
            return

        new_vertices = [self._backend.add_vertex(None) for _ in range(k)]

        M = Graph(self.get_matching())
        if M.has_edge(u, v, l):
            M.delete_edge(u, v, l)

        self._backend.del_edge(u, v, l, self._directed)
        self._backend.add_edge(u, new_vertices[0], l, self._directed)
        self._backend.add_edge(new_vertices[-1], v, l, self._directed)

        from itertools import pairwise
        self._backend.add_edges([(x, y, l) for x, y in pairwise(new_vertices)],
                                self._directed, remove_loops=True)

        if M.degree(u):
            M.add_edges([(x, y, l)
                for x, y in zip(new_vertices[::2], new_vertices[1::2])])
        else:
            M.add_edge(u, new_vertices[0], l)
            M.add_edge(new_vertices[-1], v, l)
            M.add_edges([(x, y, l)
                for x, y in zip(new_vertices[1::2], new_vertices[2::2])])

        self.update_matching(M)

    @doc_index('Overwritten methods')
    def subdivide_edges(self, edges, k):
        r"""
        Subdivide `k` times edges from an iterable container.

        For more information on the behaviour of this method, please refer to
        the documentation of
        :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.subdivide_edge`.

        .. NOTE::

            This method overwrites the
            :meth:`~sage.graphs.generic_graph.GenericGraph.subdivide_edges`
            method to ensure that resultant graph is also matching covered.

        INPUT:

        - ``edges`` -- an iterable of edges, each of which given either as
          ``(u, v)`` or ``(u, v, 'label')``

        - ``k`` -- a nonnegative even integer; common length of the
          subdivisions

        .. NOTE::

            * If the given edge is labelled with `l`, all the edges created by
              the subdivision will have the same label.

            * If no label given, the label obtained from
              :meth:`~sage.graphs.generic_graph.GenericGraph.edge_label` will
              be used. If multiple such labels are present, the first one in
              the returned list will be used.

            * The number of subdivisions must be a nonnegative even integer in
              order to ensure the resultant graph is matching covered.

            * In the context of matching covered graphs, *bisubdividing* an edge
              *t* times is defined as subdividing the edge *2t* times, for some
              nonnegative integer *t*.

            * Please note that if a single edge is present in the iterable, it
              is considered only one time for the subdivision operation.

        OUTPUT:

        - If existent edges are provided with a valid format and the parameter
          ``k`` in the argument is a nonnegative even integer then the graph is
          updated and nothing is returned, otherwise a :exc:`ValueError` is
          returned if ``k`` is not a nonnegative even integer,

        - If the graph does not contain at least one of the edges provided, a
          :exc:`ValueError` is returned. Also, a :exc:`ValueError` is thrown in case
          the provided arguments are not valid.

        EXAMPLES:

        Subdividing each edge of `K_4` some even number of times. Please note
        that the perfect matching captured at ``self.get_matching()`` also gets
        updated::

            sage: C = graphs.CycleGraph(4)
            sage: G = MatchingCoveredGraph(C)
            sage: from collections import Counter
            sage: V, E = set(G.vertices()), list(G.edges(sort=True, sort_vertices=True))
            sage: M = set(G.get_matching())
            sage: G.subdivide_edges(E, 2)
            sage: W, F = set(G.vertices()), list(G.edges(sort=True, sort_vertices=True))
            sage: N = set(G.get_matching())
            sage: sorted(W - V)
            [4, 5, 6, 7, 8, 9, 10, 11]
            sage: sorted(list((Counter(F) - Counter(E)).elements())), \
            ....: sorted(list((Counter(E) - Counter(F)).elements()))
            ([(0, 4, None), (0, 6, None), (1, 5, None), (1, 8, None),
              (2, 9, None), (2, 10, None), (3, 7, None), (3, 11, None),
              (4, 5, None), (6, 7, None), (8, 9, None), (10, 11, None)],
             [(0, 1, None), (0, 3, None), (1, 2, None), (2, 3, None)])
            sage: if (0, 1, None) in M:
            ....:     assert sorted(N - M), sorted(M - N) == \
            ....:         ([(0, 8, None), (1, 9, None), (2, 4, None), (3, 5, None), \
            ....:       (6, 7, None), (10, 11, None)], [(0, 1, None), (2, 3, None)])
            ....: else:
            ....:     assert sorted(N - M), sorted(M - N) == \
            ....:         ([(0, 6, None), (1, 10, None), (2, 11, None), (3, 7, None), \
            ....:           (4, 5, None), (8, 9, None)], [(0, 3, None), (1, 2, None)])

        If a single is present multiple times in the iterable, if that many (or
        more) edges are present in the graph, each of those edges gets subdivided::

            sage: C = graphs.CycleGraph(4)
            sage: G = MatchingCoveredGraph(C)
            sage: G.allow_multiple_edges(True)
            sage: G.add_edges([(0, 1)] * 3)
            sage: from collections import Counter
            sage: V, E = set(G.vertices()), list(G.edges(sort=True, sort_vertices=True))
            sage: M = list(G.get_matching())
            sage: G.subdivide_edges([(0, 1), (0, 1, None)], 2)
            sage: W, F = set(G.vertices()), list(G.edges(sort=True, sort_vertices=True))
            sage: N = set(G.get_matching())
            sage: sorted(W - V)
            [4, 5, 6, 7]
            sage: sorted(list((Counter(F) - Counter(E)).elements())), \
            ....: sorted(list((Counter(E) - Counter(F)).elements()))
            ([(0, 4, None), (0, 6, None), (1, 5, None), (1, 7, None),
              (4, 5, None), (6, 7, None)], [(0, 1, None), (0, 1, None)])
            sage: if (0, 1, None) in M:
            ....:     assert sorted(N - M), sorted(M - N) == \
            ....:         ([(0, 4, None), (1, 5, None)], [(0, 1, None)])
            ....:     assert sorted(N - M), sorted(M - N) == \
            ....:         ([(4, 5, None)], [])

        Subdividing edges with at least one of which is a multiple edge::

            sage: T = graphs.TricornGraph()
            sage: G = MatchingCoveredGraph(T)
            sage: G.allow_multiple_edges(True)
            sage: G.add_edges([
            ....:     (0, 1, 2), (0, 1, 3),
            ....:     (2, 3, 0.5), (2, 3, 4)
            ....: ])
            sage: G.delete_edge(0, 1, None)
            sage: from collections import Counter
            sage: V, E = set(G.vertices()), list(G.edges(sort=True, sort_vertices=True))
            sage: G.subdivide_edges([(0, 1), (1, 2), (2, 3)], 4)
            ....: # edges considered: (0, 1, 2), (1, 2, None), (2, 3, None)
            sage: W, F = set(G.vertices()), list(G.edges(sort=True, sort_vertices=True))
            sage: sorted(W - V)
            [10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21]
            sage: sorted(list((Counter(F) - Counter(E)).elements())), \
            ....: sorted(list((Counter(E) - Counter(F)).elements()))
            ([(0, 10, 2), (1, 13, 2), (1, 14, None), (2, 17, None),
              (2, 18, None), (3, 21, None), (10, 11, 2), (11, 12, 2),
              (12, 13, 2), (14, 15, None), (15, 16, None), (16, 17, None),
              (18, 19, None), (19, 20, None), (20, 21, None)],
             [(0, 1, 2), (1, 2, None), (2, 3, None)])

        Setting ``k`` to `0` does not change the graph::

            sage: H = G.copy()
            sage: G.subdivide_edges([(0, 10), (1, 13), (2, 17)], 0)
            sage: H == G and H.edges(sort=True) == G.edges(sort=True)
            True

        Subdividing edges with at least one of which is nonexistent::

            sage: G.subdivide_edges([(4, 5), (1, 5)], 4)
            ....: # edges considered: (4, 5, None), (1, 5, None)
            Traceback (most recent call last):
            ...
            ValueError: the given edge (1, 5, None) does not exist
            sage: G.subdivide_edges([(0, 4), (0, 4, None)], 4)
            Traceback (most recent call last):
            ...
            ValueError: input contains 2 copies of the edge (0, 4, None),
            but the graph contains 1

        When ``k`` is not a nonnegative even integer::

            sage: G.subdivide_edges([(0, 1), (1, 2), (2, 3)], 3)
            Traceback (most recent call last):
            ...
            ValueError: the number of subdivisions must be a nonnegative even integer, but found 3

        Providing a noniterable object as ``edges``::

            sage: G.subdivide_edges(G.order(), 4)
            Traceback (most recent call last):
            ...
            ValueError: expected an iterable of edges, but got a non-iterable object

        Providing arguments in an invalid format::

            sage: G.subdivide_edges([(0, ), (0, 1, 'label')], 4)
            Traceback (most recent call last):
            ...
            ValueError: need more than 1 value to unpack for edge: (0,)
            sage: G.subdivide_edges([(0, 1, 2, 4), (0, 1, 'label')], 4)
            Traceback (most recent call last):
            ...
            ValueError: too many values to unpack (expected 2) for edge: (0, 1, 2, 4)
            sage: G.subdivide_edges([0, (0, 1, 'label')], 4)
            Traceback (most recent call last):
            ...
            TypeError: input edge 0 is of unknown type

        .. SEEALSO::

            - :meth:`~sage.graphs.matching_covered_graph.MatchingCoveredGraph.subdivide_edge`
        """
        from collections.abc import Iterable
        if not isinstance(edges, Iterable):
            raise ValueError('expected an iterable of edges, but got a non-iterable object')

        if k < 0 or k % 2:
            raise ValueError('the number of subdivisions must be a '
                             f'nonnegative even integer, but found {k}')

        if not k:
            return

        for i, edge in enumerate(edges):
            if hasattr(edge, '__len__'):
                if len(edge) <= 1:
                    raise ValueError('need more than 1 value to unpack '
                                     f'for edge: {edge}')

                elif len(edge) > 3:
                    raise ValueError('too many values to unpack (expected 2) '
                                     f'for edge: {edge}')

            else:
                raise TypeError(f'input edge {edge} is of unknown type')

            u, v, l = None, None, None

            if len(edge) == 2:
                u, v = edge
                if not self.has_edge(u, v):
                    raise ValueError(f'the given edge {(u, v, None)} does not exist')

                l = self.edge_label(u, v)
                if self.allows_multiple_edges() and isinstance(l, list):
                    l = l[0]

            elif len(edge) == 3:
                u, v, l = edge

            edges[i] = (u, v, l)

        from collections import Counter
        edge_frequency = Counter(edges)

        for edge, n in edge_frequency.items():
            u, v, l = edge

            labels = self.edge_label(u, v)
            c = labels.count(l) if self.allows_multiple_edges() and isinstance(labels, list) else 1

            if c < n:
                raise ValueError(f'input contains {n} copies of the edge '
                                 f'{edge}, but the graph contains {c}')

        M = Graph(self.get_matching())

        from itertools import pairwise
        for i, edge in enumerate(edges):
            u, v, l = edge
            self._backend.del_edge(u, v, l, self._directed)

            new_vertices = [self._backend.add_vertex(None) for _ in range(k)]

            self._backend.add_edge(u, new_vertices[0], l, self._directed)
            self._backend.add_edge(new_vertices[-1], v, l, self._directed)
            self._backend.add_edges([(x, y, l) for x, y in pairwise(new_vertices)],
                                    self._directed, remove_loops=True)

            if M.has_edge(u, v, l):
                M.delete_edge(u, v, l)

            if M.degree(u):
                M.add_edges([(x, y, l)
                    for x, y in zip(new_vertices[::2], new_vertices[1::2])])
            else:
                M.add_edge(u, new_vertices[0], l)
                M.add_edge(new_vertices[-1], v, l)
                M.add_edges([(x, y, l)
                    for x, y in zip(new_vertices[1::2], new_vertices[2::2])])

        self.update_matching(M)

    @doc_index('Miscellaneous methods')
    def update_matching(self, matching):
        r"""
        Update the perfect matching captured in ``self._matching``.

        INPUT:

        - ``matching`` -- a perfect matching of the graph, that can be given
          using any valid input format of :class:`~sage.graphs.graph.Graph`.

        OUTPUT:

        - If ``matching`` is a valid perfect matching of the graph, then
          ``self._matching`` gets updated to this provided matching, or
          otherwise an exception is returned without any alterations to
          ``self._matching``.

        EXAMPLES:

        Providing with a valid perfect matching of the graph::

            sage: P = graphs.PetersenGraph()
            sage: G = MatchingCoveredGraph(P)
            sage: sorted(G.get_matching())
            [(0, 5, None), (1, 6, None), (2, 7, None), (3, 8, None), (4, 9, None)]
            sage: M = [(0, 1), (2, 3), (4, 9), (5, 7), (6, 8)]
            sage: G.update_matching(M)
            sage: sorted(G.get_matching())
            [(0, 1, None), (2, 3, None), (4, 9, None), (5, 7, None), (6, 8, None)]

        TESTS:

        Providing with a wrong matching::

            sage: P = graphs.PetersenGraph()
            sage: G = MatchingCoveredGraph(P)
            sage: sorted(G.get_matching())
            [(0, 5, None), (1, 6, None), (2, 7, None), (3, 8, None), (4, 9, None)]
            sage: S = str('0')
            sage: G.update_matching(S)
            Traceback (most recent call last):
            ...
            RuntimeError: the string seems corrupt: valid characters are
            ?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\]^_`abcdefghijklmnopqrstuvwxyz{|}~
            sage: T = str('graph')
            sage: G.update_matching(T)
            Traceback (most recent call last):
            ...
            RuntimeError: the string (graph) seems corrupt: for n = 40,
                the string is too short
            sage: M = Graph(G.matching())
            sage: M.add_edges([(0, 1), (0, 2)])
            sage: G.update_matching(M)
            Traceback (most recent call last):
            ...
            ValueError: the input is not a matching
            sage: N = Graph(G.matching())
            sage: N.add_edge(10, 11)
            sage: G.update_matching(N)
            Traceback (most recent call last):
            ...
            ValueError: the input is not a matching of the graph
            sage: J = Graph()
            sage: J.add_edges([(0, 1), (2, 3)])
            sage: G.update_matching(J)
            Traceback (most recent call last):
            ...
            ValueError: the input is not a perfect matching of the graph
        """
        try:
            M = Graph(matching)

            if any(d != 1 for d in M.degree()):
                raise ValueError("the input is not a matching")

            if any(not self.has_edge(edge) for edge in M.edge_iterator()):
                raise ValueError("the input is not a matching of the graph")

            if (self.order() != M.order()):
                raise ValueError("the input is not a perfect matching of the graph")

            self._matching = M.edges()

        except Exception as exception:
            raise exception


__doc__ = __doc__.replace('{INDEX_OF_METHODS}', gen_thematic_rest_table_index(MatchingCoveredGraph, only_local_functions=False))
