##
## Calculates the orbits of the group generated by gens on the set
## setup.coords. Stores the representatives of the orbits in setup.orbitreps
## and the elements that map the representatives to the elements of setup.coords
## in setup.conjelts.
##
## <gens> must be (signed) permutations in list format.
##

InstallGlobalFunction(MAJORANA_Orbits,

    function(gens, setup)

    local   conjelts,
            orbitreps,
            i, j,
            orb,
            gen,
            elts,
            count,
            dim,
            p,
            q,
            h,
            g;

    dim := Size(setup.coords);

    # The list <orbitreps> form the representative elements of the orbits
    # and the list <conjelts> contains signed permutations (as lists) that map
    # the representatives to the elements of setup.coords
    conjelts := [1..dim]*0;
    orbitreps := [];

    # For all elements of rep.setup.coords
    for i in [1..dim] do
        # If the element is not yet in an orbit
        if conjelts[i] = 0 then

            # Add a new orbit
            Add(orbitreps, i);
            conjelts[i] := [1..dim];

            # Store the orbit and the group elements that map the
            # representative of this orbit to its elements. The integer
            # <count> keeps track of where we are in the elts list.
            orb := [i];
            elts := [[1..dim]];
            count := 0;

            # For anything already in the orbit
            for p in orb do

                count := count + 1;
                h := elts[count];

                # Apply the generators of the group to q
                for gen in gens do
                    q := gen[p];

                    if q < 0 then q := -q; fi;

                    # If this element has not already been added to the orbit then
                    # do so now, and also store the corresponding group elt
                    if conjelts[q] = 0 then

                        # Multiply h and gen
                        g := [];
                        for j in h do
                            if j > 0 then
                                Add(g, gen[j]);
                            else
                                Add(g, -gen[-j]);
                            fi;
                        od;

                        Add(orb, q);
                        Add(elts, g);
                        conjelts[q] := g;
                    fi;
                od;
            od;
        fi;
    od;

    return rec( conjelts := conjelts,
                orbitreps := orbitreps  );

    end );

##
## Calculates the orbitals of the group generated by gens on the set
## setup.coords. Stores the representatives of the orbitals in setup.pairreps
## and the elements that map the representatives to the pairs of elts of setup.coords
## in setup.pairconjelts.
##
## <gens> must be (signed) permutations in list format.
##
## The argument <t> is an integer. If it is greater than 0 then it indicates that
## the orbitals have already been found on the first <t> elements of setup.coords.
##

InstallGlobalFunction(MAJORANA_Orbitals,

    function(gens,t,setup)

    local   dim, i, j;

    dim := Size(setup.coords);

    # Loop over pairs of elements of setup.coords
    for i in [1..dim] do
        for j in [Maximum(i,t + 1)..dim] do

            # If this pair is not already in an orbit
            if setup.pairorbit[i, j] = 0 then

                # Find the orbital containing this pair
                MAJORANA_NewOrbital([i,j], gens, setup);
            fi;
        od;
    od;

    end );

##
## Computes the orbital under the action of the group generated by <gens> that
## contains the pair <pnt>.
##
## <gens> must be (signed) permutations in list format.
##

InstallGlobalFunction( MAJORANA_NewOrbital,

    function( pnt, gens, setup)

    local orb, elts, count, y, p, h, q, z, g, i, pos, sign, gen;

    # Added <pnt> as the orbital representative
    Add(setup.pairreps, pnt);

    # Store the orbital and the group elements that map the
    # representative of this orbital to its elements. The integer
    # <count> keeps track of where we are in the elts list.
    orb := [ pnt ];
    elts := [ [1 .. Size(setup.coords)] ];
    count := 0;

    # Store the pairorbit and pairconj data for this representative element
    y := Size(setup.pairreps);
    setup.pairorbit[pnt[1], pnt[2]] := y;
    setup.pairorbit[pnt[2], pnt[1]] := y;

    setup.pairconj[pnt[1], pnt[2]] := 1;
    setup.pairconj[pnt[2], pnt[1]] := 1;

    # For anything already in the orbital
    for p in orb do

        count := count + 1;
        h := elts[count];

        # Apply the generators of the group to q
        for gen in gens do

            q := gen{p};

            if q[1] < 0 then q[1] := -q[1]; fi;
            if q[2] < 0 then q[2] := -q[2]; fi;

            z := setup.pairorbit[q[1], q[2]];

            # If this element has not already been added to the orbital then
            # do so now, and also store the corresponding data
            if z = 0 then

                # Multiply h and gen
                g := [];
                for i in h do
                    if i > 0 then
                        Add(g, gen[i]);
                    else
                        Add(g, -gen[-i]);
                    fi;
                od;

                # Add the pair and the group element to the orbital and elt list
                Add( orb, q );
                Add( elts, g);

                # Store the pair orbit data (adjusting for sign)
                if Product(g{orb[1]}) < 0 then
                    sign := -1;
                else
                    sign := 1;
                fi;

                setup.pairorbit[q[1], q[2]] := sign*y;
                setup.pairorbit[q[2], q[1]] := sign*y;

                # Store the pair conj data (adding g to pairconjelts if necessary)
                pos := Position(setup.pairconjelts, g);

                if pos = fail then
                    Add(setup.pairconjelts, g);
                    pos := Size(setup.pairconjelts);
                fi;

                setup.pairconj[q[1], q[2]] := pos;
                setup.pairconj[q[2], q[1]] := pos;
            fi;
        od;
    od;

    return;

    end );
